# NAME

Getopt::EX::Hashed - Getopt::Long을 위한 해시 객체 자동화

# VERSION

Version 1.07

# SYNOPSIS

    # script/foo
    use App::foo;
    App::foo->new->run();

    # lib/App/foo.pm
    package App::foo;

    use Getopt::EX::Hashed; {
        Getopt::EX::Hashed->configure( DEFAULT => [ is => 'rw' ] );
        has start    => ' =i  s begin ' , default => 1;
        has end      => ' =i  e       ' ;
        has file     => ' =s@ f       ' , any => qr/^(?!\.)/;
        has score    => ' =i          ' , min => 0, max => 100;
        has answer   => ' =i          ' , must => sub { $_[1] == 42 };
        has mouse    => ' =s          ' , any => [ 'Frankie', 'Benjy' ];
        has question => ' =s          ' , any => qr/^(life|universe|everything)$/i;
    } no Getopt::EX::Hashed;

    sub run {
        my $app = shift;
        use Getopt::Long;
        $app->getopt or pod2usage();
        if ($app->answer == 42) {
            $app->question //= 'life';
            ...

# DESCRIPTION

**Getopt::EX::Hashed**는 **Getopt::Long** 및 **Getopt::EX::Long**을 포함한 호환 모듈용으로, 명령줄 옵션 값을 저장할 해시 객체의 생성을 자동화하는 모듈입니다. 모듈 이름은 **Getopt::EX** 접두사를 공유하지만, 지금까지는 **Getopt::EX**의 다른 모듈과 독립적으로 동작합니다.

이 모듈의 주요 목적은 초기화와 명세를 한 곳에 통합하는 것입니다. 또한 간단한 검증 인터페이스를 제공합니다.

`is` 매개변수가 주어지면 접근자 메서드가 자동으로 생성됩니다. 동일한 함수가 이미 정의되어 있으면 프로그램은 치명적인 오류를 발생시킵니다. 객체가 파기된 후에도 접근자는 계속 남습니다. 파기 시에 이를 제거하고 싶다면 `REMOVE_ACCESSOR` 구성 매개변수를 true로 설정하십시오.

# FUNCTION

## **has**

옵션 매개변수는 다음 형태로 선언하십시오. 괄호는 명확성을 위한 것이며 생략할 수 있습니다.

    has option_name => ( param => value, ... );

예를 들어, 정수 값을 매개변수로 받는 옵션 `--number`를 정의하고, `-n`으로도 사용할 수 있게 하려면 다음과 같이 하십시오

    has number => spec => "=i n";

접근자는 첫 번째 이름으로 생성됩니다. 이 예에서 접근자는 `$app->number`로 정의됩니다.

배열 참조가 주어지면 여러 이름을 한 번에 선언할 수 있습니다.

    has [ 'left', 'right' ] => ( spec => "=i" );

이름이 플러스(`+`)로 시작하면, 주어진 매개변수는 기존 설정을 갱신합니다.

    has '+left' => ( default => 1 );

`spec` 매개변수의 경우, 라벨은 그것이 첫 번째 매개변수라면 생략할 수 있습니다.

    has left => "=i", default => 1;

매개변수 개수가 홀수이면, 첫 번째 매개변수는 암시적 라벨이 있는 것으로 처리됩니다: 코드 참조이면 `action`, 그렇지 않으면 `spec`.

다음 매개변수를 사용할 수 있습니다.

- \[ **spec** => \] _string_

    옵션 명세를 제공합니다. `spec =>` 라벨은 첫 번째 매개변수일 때에만 생략할 수 있습니다.

    _string_에서는 옵션 명세와 별칭 이름이 공백으로 구분되며, 어떤 순서로든 나타날 수 있습니다.

    정수를 값으로 받는 `--start`라는 옵션을 만들고, `-s`와 `--begin`이라는 이름으로도 사용할 수 있게 하려면 다음과 같이 선언하십시오.

        has start => "=i s begin";

    위 선언은 다음 문자열로 컴파일됩니다.

        start|s|begin=i

    이는 `Getopt::Long` 정의에 부합합니다. 물론 다음과 같이 작성할 수도 있습니다:

        has start => "s|begin=i";

    이름과 별칭에 밑줄(`_`)이 포함되어 있으면, 밑줄 대신 대시(`-`)를 사용한 또 다른 별칭 이름이 정의됩니다.

        has a_to_z => "=s";

    위 선언은 다음 문자열로 컴파일됩니다.

        a_to_z|a-to-z=s

    옵션 사양이 필요하지 않다면, 값으로 빈 문자열(또는 공백만 있는 문자열)을 주십시오. 사양 문자열이 없으면 해당 멤버는 옵션으로 취급되지 않습니다.

- **alias** => _string_

    추가 별칭 이름은 **alias** 매개변수로도 지정할 수 있습니다. `spec` 매개변수에 있는 것과 차이가 없습니다.

        has start => "=i", alias => "s begin";

- **is** => `ro` | `rw`

    접근자 메서드를 만들려면 `is` 매개변수가 필요합니다. 값으로 읽기 전용은 `ro`, 읽기/쓰기는 `rw`를 설정하십시오.

    읽기/쓰기 접근자는 lvalue 속성을 가지므로 대입할 수 있습니다. 다음과 같이 사용할 수 있습니다:

        $app->foo //= 1;

    이는 다음과 같이 작성하는 것보다 훨씬 간단합니다.

        $app->foo(1) unless defined $app->foo;

    이후 모든 멤버에 대해 접근자를 만들고 싶다면, `configure`를 사용해 `DEFAULT` 매개변수를 설정하십시오.

        Getopt::EX::Hashed->configure( DEFAULT => [ is => 'rw' ] );

    할당 가능한 접근자가 마음에 들지 않는다면 `ACCESSOR_LVALUE` 파라미터를 0으로 설정하세요. 접근자는 `new` 시점에 생성되므로, 이 값은 모든 멤버에 대해 유효합니다.

- **default** => _value_ | _coderef_

    기본값을 설정합니다. 기본값이 주어지지 않으면 멤버는 `undef`로 초기화됩니다.

    값이 ARRAY 또는 HASH에 대한 참조인 경우, 각 `new` 호출마다 얕은 복사본이 생성됩니다. 이는 참조 자체는 복사되지만, 내용은 공유된다는 뜻입니다. 배열이나 해시의 내용을 수정하면 모든 인스턴스에 영향을 미칩니다.

    코드 참조가 주어지면 기본값을 얻기 위해 **new** 시점에 호출됩니다. 이는 선언 시점이 아니라 실행 시점에 값을 평가하고자 할 때 효과적입니다. 기본 동작을 정의하려면 **action** 파라미터를 사용하세요. 코드 참조를 초기값으로 설정하려면, 코드 참조를 반환하는 코드 참조를 지정해야 합니다.

    SCALAR에 대한 참조가 주어지면, 옵션 값은 해시 객체 멤버가 아니라 해당 참조가 가리키는 데이터에 저장됩니다. 이 경우 해시 멤버에 접근해도 기대하는 값을 얻을 수 없습니다.

- \[ **action** => \] _coderef_

    파라미터 `action`은 옵션을 처리하기 위해 호출되는 코드 참조를 받습니다. `action =>` 라벨은 오직 첫 번째 파라미터일 때에만 생략할 수 있습니다.

    호출 시, 해시 객체가 `$_`로 전달됩니다.

        has [ qw(left right both) ] => '=i';
        has "+both" => sub {
            $_->{left} = $_->{right} = $_[1];
        };

    비옵션 인수를 처리하기 위해 `"<>"`에 대해 이것을 사용할 수 있습니다. 이 경우 spec 매개변수는 중요하지 않으며 필요하지 않습니다.

        has ARGV => default => [];
        has "<>" => sub {
            push @{$_->{ARGV}}, $_[0];
        };

다음 파라미터들은 모두 데이터 검증을 위한 것입니다. 먼저, `must`는 범용 검증기로서 무엇이든 구현할 수 있습니다. 나머지는 일반 규칙에 대한 단축형입니다.

- **must** => _coderef_ | \[ _coderef_ ... \]

    파라미터 `must`는 옵션 값을 검증하기 위한 코드 참조를 받습니다. `action`과 동일한 인수를 취하고 불리언을 반환합니다. 다음 예시에서 옵션 **--answer**는 유효한 값으로 오직 42만 허용합니다.

        has answer => '=i',
            must => sub { $_[1] == 42 };

    코드 참조가 여러 개 주어지면, 모든 코드가 참을 반환해야 합니다.

        has answer => '=i',
            must => [ sub { $_[1] >= 42 }, sub { $_[1] <= 42 } ];

- **min** => _number_
- **max** => _number_

    인자의 최소 및 최대 한계를 설정합니다.

- **any** => _arrayref_ | qr/_regex_/ | _coderef_

    유효한 문자열 파라미터 리스트를 설정합니다. 각 항목은 문자열, 정규식 참조, 또는 코드 참조가 될 수 있습니다. 인자는 주어진 리스트의 항목과 동일하거나 일치할 때 유효합니다. 값이 arrayref가 아니면 단일 항목 리스트로 간주됩니다(보통 regexpref 또는 coderef).

    다음 선언들은 거의 동등하며, 두 번째 것은 대소문자를 구분하지 않습니다.

        has question => '=s',
            any => [ 'life', 'universe', 'everything' ];

        has question => '=s',
            any => qr/^(life|universe|everything)$/i;

    선택적 인자를 사용 중이라면, 기본값을 리스트에 포함하는 것을 잊지 마세요. 그렇지 않으면 검증 오류가 발생합니다.

        has question => ':s',
            any => [ 'life', 'universe', 'everything', '' ];

# METHOD

## **new**

새 해시 객체를 생성하는 클래스 메서드입니다. 모든 멤버를 기본값으로 초기화하고 설정에 따라 접근자 메서드를 생성합니다. blessed 해시 참조를 반환합니다. LOCK\_KEYS가 활성화된 경우 해시 키는 잠깁니다.

## **optspec**

`GetOptions` 함수에 전달할 수 있는 옵션 명세 리스트를 반환합니다.

    GetOptions($obj->optspec)

`GetOptions`는 첫 번째 인수로 해시 참조를 제공하여 값을 해시에 저장하는 기능이 있지만, 반드시 그럴 필요는 없습니다.

## **getopt** \[ _arrayref_ \]

호출자의 컨텍스트에서 정의된 적절한 함수를 호출하여 옵션을 처리합니다.

    $obj->getopt

    $obj->getopt(\@argv);

위의 예제들은 다음 코드에 대한 단축형입니다.

    GetOptions($obj->optspec)

    GetOptionsFromArray(\@argv, $obj->optspec)

## **use\_keys** _keys_

LOCK\_KEYS가 활성화되면, 존재하지 않는 멤버에 접근할 경우 오류가 발생합니다. 접근하기 전에 새로운 멤버 키를 선언하려면 이 메서드를 사용하십시오.

    $obj->use_keys( qw(foo bar) );

임의의 키에 접근하려면 객체의 잠금을 해제하세요.

    use Hash::Util 'unlock_keys';
    unlock_keys %{$obj};

`LOCK_KEYS` 매개변수와 함께 `configure`로 이 동작을 변경할 수 있습니다.

## **configure** **label** => _value_, ...

객체를 생성하기 전에 클래스 메서드 `Getopt::EX::Hashed->configure()`를 사용하십시오. 이 정보는 호출하는 각 패키지마다 별도로 저장됩니다. `new()`를 호출한 후, 패키지 수준 구성은 객체가 사용하도록 객체에 복사됩니다. 객체 수준 구성을 업데이트하려면 `$obj->configure()`를 사용하십시오.

다음 구성 매개변수를 사용할 수 있습니다.

- **LOCK\_KEYS** (default: 1)

    해시 키를 잠급니다. 이는 오타나 기타 실수로 의도치 않은 해시 항목이 생성되는 것을 방지합니다.

- **REPLACE\_UNDERSCORE** (default: 1)

    밑줄을 대시로 바꾼 옵션 별칭을 자동으로 생성합니다.

- **REMOVE\_UNDERSCORE** (default: 0)

    밑줄을 제거한 옵션 별칭을 자동으로 생성합니다.

- **GETOPT** (default: 'GetOptions')
- **GETOPT\_FROM\_ARRAY** (default: 'GetOptionsFromArray')

    `getopt` 메서드에서 호출되는 함수 이름을 설정합니다.

- **ACCESSOR\_PREFIX** (default: '')

    지정하면, 접근자 메서드를 만들기 위해 멤버 이름 앞에 접두사가 추가됩니다. `ACCESSOR_PREFIX`가 `opt_`로 정의되어 있으면, 멤버 `file`의 접근자는 `opt_file`이 됩니다.

- **ACCESSOR\_LVALUE** (default: 1)

    참이면, 읽기-쓰기 접근자는 lvalue 속성을 가집니다. 그러한 동작이 마음에 들지 않으면 0으로 설정하세요.

- **REMOVE\_ACCESSOR** (default: 0)

    true이면, 객체가 파기될 때 패키지 네임스페이스에서 접근자 메서드가 제거됩니다. 이는 모듈이 기존 클래스에 내장되어 있고, 더 이상 필요하지 않을 때 생성된 접근자가 남지 않도록 해야 하는 경우를 위한 것입니다. 기본적으로 접근자는 그대로 유지됩니다.

- **DEFAULT**

    기본 매개변수를 설정합니다. `has`가 호출될 때 DEFAULT 매개변수가 명시적 매개변수 앞에 삽입됩니다. 둘 모두에 매개변수가 나타나면, 명시적인 것이 우선합니다. `+`가 있는 누적 호출에는 영향을 주지 않습니다.

    DEFAULT의 전형적인 사용은 `is`로 이후의 모든 해시 항목에 대한 접근자 메서드를 준비하는 것입니다. 재설정하려면 `DEFAULT => []`를 선언하세요.

        Getopt::EX::Hashed->configure(DEFAULT => [ is => 'ro' ]);

## **reset**

클래스를 원래 상태로 재설정합니다.

# SEE ALSO

[Getopt::Long](https://metacpan.org/pod/Getopt%3A%3ALong)

[Getopt::EX](https://metacpan.org/pod/Getopt%3A%3AEX), [Getopt::EX::Long](https://metacpan.org/pod/Getopt%3A%3AEX%3A%3ALong)

# AUTHOR

Kazumasa Utashiro

# COPYRIGHT

The following copyright notice applies to all the files provided in
this distribution, including binary files, unless explicitly noted
otherwise.

Copyright 2021-2026 Kazumasa Utashiro

# LICENSE

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.
