<?php
/**
 * @license GPL-2.0-or-later
 * @file
 */

declare( strict_types=1 );

namespace Wikimedia\Stats;

use TypeError;
use Wikimedia\Stats\Metrics\MetricInterface;
use Wikimedia\Stats\Metrics\NullMetric;

/**
 * Singleton cache for Metric instances.
 *
 * For reuse and collision avoidance.  Serves as the data source for Metric Emitters.
 *
 * @author Cole White
 * @since 1.41
 */
class StatsCache {

	public const DELIMITER = '.';

	/** @var MetricInterface[] */
	private array $cache = [];

	/**
	 * Get a metric object from cache or null.
	 *
	 * @param string $component
	 * @param string $name
	 * @param string $expectedClass
	 * @return MetricInterface|null
	 * @throws TypeError If cached value is for a different metric type.
	 */
	public function get( string $component, string $name, string $expectedClass ) {
		$key = self::cacheKey( $component, $name );
		$metric = $this->cache[$key] ?? null;
		if ( $metric !== null && get_class( $metric ) !== $expectedClass ) {
			throw new TypeError( "Encountered metric name collision: $key defined as "
				. get_class( $metric ) . " but $expectedClass was requested" );
		}
		return $metric;
	}

	/**
	 * Add a metric object to the cache.
	 *
	 * @param string $component
	 * @param string $name
	 * @param MetricInterface|NullMetric $metric
	 */
	public function set( string $component, string $name, $metric ): void {
		$this->cache[self::cacheKey( $component, $name )] = $metric;
	}

	/**
	 * Get all metrics from cache.
	 *
	 * @return MetricInterface[]
	 */
	public function getAllMetrics(): array {
		return $this->cache;
	}

	/**
	 * Clears the cache.
	 */
	public function clear(): void {
		$this->cache = [];
	}

	/**
	 * Get the metric formatted name.
	 *
	 * Takes the provided name and constructs a more specific name by combining
	 * component and name.
	 */
	public static function cacheKey( string $component, string $name ): string {
		// mitigate collision of empty-component metric with a component metric
		if ( $component !== '' ) {
			return implode( self::DELIMITER, [ $component, $name ] );
		}
		return $name;
	}
}
