/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * vim: set ts=8 sts=2 et sw=2 tw=80:
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/* JSClass definition and its component types, plus related interfaces. */

#ifndef js_Class_h
#define js_Class_h

#include "mozilla/Attributes.h"
#include "mozilla/Maybe.h"

#include "jstypes.h"

#include "js/CallArgs.h"
#include "js/HeapAPI.h"
#include "js/Id.h"
#include "js/TypeDecls.h"

/*
 * A JSClass acts as a vtable for JS objects that allows JSAPI clients to
 * control various aspects of the behavior of an object like property lookup.
 * It contains some engine-private extensions that allows more control over
 * object behavior and, e.g., allows custom slow layout.
 */

struct JSAtomState;
struct JSFunctionSpec;

namespace js {

class PropertyResult;

// These are equal to js::FunctionClass / js::ExtendedFunctionClass.
extern JS_PUBLIC_DATA const JSClass* const FunctionClassPtr;
extern JS_PUBLIC_DATA const JSClass* const FunctionExtendedClassPtr;

}  // namespace js

namespace JS {

/**
 * Per ES6, the [[DefineOwnProperty]] internal method has three different
 * possible outcomes:
 *
 * -   It can throw an exception (which we indicate by returning false).
 *
 * -   It can return true, indicating unvarnished success.
 *
 * -   It can return false, indicating "strict failure". The property could
 *     not be defined. It's an error, but no exception was thrown.
 *
 * It's not just [[DefineOwnProperty]]: all the mutating internal methods have
 * the same three outcomes. (The other affected internal methods are [[Set]],
 * [[Delete]], [[SetPrototypeOf]], and [[PreventExtensions]].)
 *
 * If you think this design is awful, you're not alone.  But as it's the
 * standard, we must represent these boolean "success" values somehow.
 * ObjectOpSuccess is the class for this. It's like a bool, but when it's false
 * it also stores an error code.
 *
 * Typical usage:
 *
 *     ObjectOpResult result;
 *     if (!DefineProperty(cx, obj, id, ..., result)) {
 *         return false;
 *     }
 *     if (!result) {
 *         return result.reportError(cx, obj, id);
 *     }
 *
 * Users don't have to call `result.report()`; another possible ending is:
 *
 *     argv.rval().setBoolean(result.ok());
 *     return true;
 */
class ObjectOpResult {
 private:
  /**
   * code_ is either one of the special codes OkCode or Uninitialized, or an
   * error code. For now the error codes are JS friend API and are defined in
   * js/public/friend/ErrorNumbers.msg.
   *
   * code_ is uintptr_t (rather than uint32_t) for the convenience of the
   * JITs, which would otherwise have to deal with either padding or stack
   * alignment on 64-bit platforms.
   */
  uintptr_t code_;

 public:
  enum SpecialCodes : uintptr_t { OkCode = 0, Uninitialized = uintptr_t(-1) };

  ObjectOpResult() : code_(Uninitialized) {}

  /* Return true if succeed() was called. */
  bool ok() const {
    MOZ_ASSERT(code_ != Uninitialized);
    return code_ == OkCode;
  }

  explicit operator bool() const { return ok(); }

  /* Set this ObjectOpResult to true and return true. */
  bool succeed() {
    code_ = OkCode;
    return true;
  }

  /*
   * Set this ObjectOpResult to false with an error code.
   *
   * Always returns true, as a convenience. Typical usage will be:
   *
   *     if (funny condition) {
   *         return result.fail(JSMSG_CANT_DO_THE_THINGS);
   *     }
   *
   * The true return value indicates that no exception is pending, and it
   * would be OK to ignore the failure and continue.
   */
  bool fail(uint32_t msg) {
    MOZ_ASSERT(msg != OkCode);
    code_ = msg;
    return true;
  }

  JS_PUBLIC_API bool failCantRedefineProp();
  JS_PUBLIC_API bool failReadOnly();
  JS_PUBLIC_API bool failGetterOnly();
  JS_PUBLIC_API bool failCantDelete();

  JS_PUBLIC_API bool failCantSetInterposed();
  JS_PUBLIC_API bool failCantDefineWindowElement();
