//! Array support types

use crate::avm2::value::Value;
use gc_arena::Collect;
use std::collections::BTreeMap;

const MIN_SPARSE_LENGTH: usize = 32;
const MAX_DENSE_LENGTH: usize = 1 << 28;

/// The array storage portion of an array object.
///
/// Array values may consist of either standard `Value`s or "holes": values
/// which are not properties of the associated object and must be resolved in
/// the prototype.
#[derive(Clone, Collect)]
#[collect(no_drop)]
pub enum ArrayStorage<'gc> {
    /// Dense arrays store a vector of values and a count of non-holes in the vector (m_denseUsed in avmplus).
    Dense {
        storage: Vec<Option<Value<'gc>>>,
        occupied_count: usize,
    },
    /// Sparse arrays store a BTreeMap of values and an explicit ECMAScript "Array.length".
    Sparse {
        storage: BTreeMap<usize, Value<'gc>>,
        length: usize,
    },
}

/// An iterator over array storage. This iterator will yield `Some(None)` for holes.
struct ArrayStorageIterator<'a, 'gc> {
    storage: &'a ArrayStorage<'gc>,
    index: usize,
    index_back: usize,
}

impl<'gc> Iterator for ArrayStorageIterator<'_, 'gc> {
    type Item = Option<Value<'gc>>;

    fn next(&mut self) -> Option<Self::Item> {
        if self.index >= self.index_back {
            None
        } else {
            let value = match &self.storage {
                ArrayStorage::Dense { storage, .. } => storage[self.index],
                ArrayStorage::Sparse { storage, .. } => storage.get(&self.index).copied(),
            };
            self.index += 1;
            Some(value)
        }
    }
}

impl DoubleEndedIterator for ArrayStorageIterator<'_, '_> {
    fn next_back(&mut self) -> Option<Self::Item> {
        if self.index >= self.index_back || self.index_back == 0 {
            None
        } else {
            self.index_back -= 1;
            let value = match &self.storage {
                ArrayStorage::Dense { storage, .. } => storage[self.index_back],
                ArrayStorage::Sparse { storage, .. } => storage.get(&self.index_back).copied(),
            };
            Some(value)
        }
    }
}

impl ExactSizeIterator for ArrayStorageIterator<'_, '_> {
    fn len(&self) -> usize {
        self.index_back - self.index
    }
}

impl<'gc> ArrayStorage<'gc> {
    /// Construct new array storage.
    ///
    /// The length parameter indicates how big the array storage should start
    /// out as. All array storage consists of holes.
    pub fn new(length: usize) -> Self {
        if length > MAX_DENSE_LENGTH {
            ArrayStorage::Sparse {
                storage: BTreeMap::new(),
                length,
            }
        } else {
            ArrayStorage::Dense {
                storage: Vec::with_capacity(length),
                occupied_count: 0,
            }
        }
    }

    /// Convert a set of arguments into array storage.
    pub fn from_args(values: &[Value<'gc>]) -> Self {
        Self::from_iter(values.iter().copied())
    }

    /// Wrap an existing storage Vec in an `ArrayStorage`.
    pub fn from_storage(storage: Vec<Option<Value<'gc>>>) -> Self {
        let occupied_count = storage.iter().filter(|v| v.is_some()).count();

        ArrayStorage::Dense {
            storage,
            occupied_count,
        }
    }

    /// Replace the existing dense storage with a new dense storage.
    /// Panics if this `ArrayStorage` is sparse.
    pub fn replace_dense_storage(&mut self, new_storage: Vec<Option<Value<'gc>>>) {
        let new_occupied_count = new_storage.iter().filter(|v| v.is_some()).count();

        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                *occupied_count = new_occupied_count;
                *storage = new_storage;
            }
            ArrayStorage::Sparse { .. } => {
                panic!("Cannot replace dense storage on sparse ArrayStorage");
            }
        }
    }

    /// Retrieve a value from array storage by index.
    ///
    /// Array holes and out of bounds values will be treated the same way, by
    /// yielding `None`.
    pub fn get(&self, item: usize) -> Option<Value<'gc>> {
        match &self {
            ArrayStorage::Dense { storage, .. } => storage.get(item).copied().unwrap_or(None),
            ArrayStorage::Sparse { storage, .. } => storage.get(&item).copied(),
        }
    }

    /// Get the next index after the given index that contains a value.
    pub fn get_next_enumerant(&self, last_index: usize) -> Option<usize> {
        match &self {
            ArrayStorage::Dense { storage, .. } => {
                let mut last_index = last_index;
                while last_index < storage.len() {
                    if storage[last_index].is_some() {
                        return Some(last_index + 1);
                    }
                    last_index += 1;
                }
                None
            }
            ArrayStorage::Sparse { storage, .. } => {
                if let Some((&key, _)) = storage.range(last_index..).next() {
                    return Some(key + 1);
                }
                None
            }
        }
    }

    /// Set an array storage slot to a particular value.
    ///
    /// If the item index extends beyond the length of the array, then the
    /// array will be extended with holes.
    pub fn set(&mut self, item: usize, value: Value<'gc>) {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                if item >= storage.len() {
                    if Self::should_convert_to_sparse(item + 1, *occupied_count) {
                        self.convert_to_sparse();
                        if let ArrayStorage::Sparse { storage, length } = self {
                            *length = item + 1;
                            storage.insert(item, value);
                        }
                    } else {
                        storage.resize(item + 1, None);
                        storage[item] = Some(value);
                        *occupied_count += 1;
                    }
                } else {
                    if storage[item].is_none() {
                        *occupied_count += 1;
                    }
                    storage[item] = Some(value);
                }
            }
            ArrayStorage::Sparse { storage, length } => {
                storage.insert(item, value);
                if item >= *length {
                    *length = item + 1;
                }
            }
        }
    }

    /// Delete an array storage slot, leaving a hole.
    pub fn delete(&mut self, item: usize) {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                if let Some(i) = storage.get_mut(item) {
                    if i.is_some() {
                        *i = None;
                        *occupied_count -= 1;
                        self.maybe_convert_to_sparse();
                    }
                }
            }
            ArrayStorage::Sparse { storage, .. } => {
                storage.remove(&item);
                self.maybe_convert_to_dense();
            }
        }
    }

    /// Get the length of the array.
    pub fn length(&self) -> usize {
        match &self {
            ArrayStorage::Dense { storage, .. } => storage.len(),
            ArrayStorage::Sparse { length, .. } => *length,
        }
    }

    /// Set the length of the array.
    pub fn set_length(&mut self, size: usize) {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                // Update occupied_count
                if let Some(slice) = storage.get(size..) {
                    // We need to update occupied_count only when shrinking the array.
                    //
                    // Now we need to choose between iterating over the deleted elements,
                    // or the rest.  It's doesn't matter for correctness, so let's pick
                    // the smaller set for performance.
                    if slice.len() <= storage.len() / 2 {
                        let occupied_to_delete = slice.iter().filter(|v| v.is_some()).count();
                        *occupied_count = occupied_count.saturating_sub(occupied_to_delete);
                    } else {
                        let occupied_to_set =
                            storage[0..size].iter().filter(|v| v.is_some()).count();
                        *occupied_count = occupied_to_set;
                    }
                }

                if Self::should_convert_to_sparse(size, *occupied_count) {
                    self.convert_to_sparse();
                    if let ArrayStorage::Sparse { .. } = self {
                        self.set_length(size);
                    }
                } else {
                    storage.resize(size, None);
                }
            }
            ArrayStorage::Sparse { storage, length } => {
                if size > *length {
                    *length = size;
                } else {
                    storage.retain(|&k, _| k < size);
                    *length = size;
                    self.maybe_convert_to_dense();
                }
            }
        }
    }

    /// Append the contents of another array into this one.
    ///
    /// The values in the other array remain there and are merely copied into
    /// this one.
    ///
    /// Holes are copied as holes and not resolved at append time.
    pub fn append(&mut self, other_array: &Self) {
        for value in other_array.iter() {
            if let Some(value) = value {
                self.push(value);
            } else {
                self.push_hole();
            }
        }
    }

    /// Push a single value onto the end of this array.
    ///
    /// It is not possible to push a hole onto the array.
    pub fn push(&mut self, item: Value<'gc>) {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                storage.push(Some(item));
                *occupied_count += 1;
            }
            ArrayStorage::Sparse { storage, length } => {
                storage.insert(*length, item);
                *length += 1;
            }
        }
    }

    /// Determine if the array should be converted to a sparse representation based on its size and the number of occupied slots.
    fn should_convert_to_sparse(size: usize, occupied_count: usize) -> bool {
        (occupied_count < (size / 4) && size > MIN_SPARSE_LENGTH) || size > MAX_DENSE_LENGTH
    }

    /// Convert the array storage to a sparse representation.
    fn convert_to_sparse(&mut self) {
        if let ArrayStorage::Dense { storage, .. } = self {
            let mut new_storage = BTreeMap::new();
            for (i, v) in storage.iter().enumerate() {
                if let Some(v) = v {
                    new_storage.insert(i, *v);
                }
            }
            *self = ArrayStorage::Sparse {
                storage: new_storage,
                length: storage.len(),
            };
        }
    }

    /// Convert the array to a sparse representation if it meets the criteria.
    fn maybe_convert_to_sparse(&mut self) {
        if let ArrayStorage::Dense {
            storage,
            occupied_count,
        } = self
        {
            if Self::should_convert_to_sparse(storage.len(), *occupied_count) {
                self.convert_to_sparse();
            }
        }
    }

    /// Convert the array to a dense representation if it meets the criteria.
    fn maybe_convert_to_dense(&mut self) {
        if let ArrayStorage::Sparse { storage, length } = self {
            if storage.is_empty() && *length == 0 {
                *self = ArrayStorage::Dense {
                    storage: Vec::new(),
                    occupied_count: 0,
                };
            }
        }
    }

    /// Push an array hole onto the end of this array.
    pub fn push_hole(&mut self) {
        match self {
            ArrayStorage::Dense { storage, .. } => {
                storage.push(None);
                self.maybe_convert_to_sparse();
            }
            ArrayStorage::Sparse { length, .. } => {
                *length += 1;
            }
        }
    }

    /// Pop a value from the back of the array.
    ///
    /// This method preferentially pops non-holes from the array first. If a
    /// hole is popped, it will become `undefined`.
    pub fn pop(&mut self) -> Value<'gc> {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                let non_hole = storage
                    .iter()
                    .enumerate()
                    .rposition(|(_, item)| item.is_some());

                if let Some(non_hole) = non_hole {
                    *occupied_count -= 1;
                    let value = storage.remove(non_hole).unwrap();
                    self.maybe_convert_to_sparse();
                    value
                } else {
                    storage.pop().unwrap_or(None).unwrap_or(Value::Undefined)
                }
            }
            ArrayStorage::Sparse { storage, length } => {
                let non_hole = storage.pop_last();
                if let Some((_index, value)) = non_hole {
                    *length -= 1;
                    value
                } else {
                    if *length > 0 {
                        *length -= 1;
                    }
                    self.maybe_convert_to_dense();
                    Value::Undefined
                }
            }
        }
    }

    /// Shift a value from the front of the array.
    pub fn shift(&mut self) -> Value<'gc> {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                if !storage.is_empty() {
                    let value = storage.remove(0);
                    if value.is_some() {
                        *occupied_count -= 1;
                    }
                    self.maybe_convert_to_sparse();
                    return value.unwrap_or(Value::Undefined);
                }
                Value::Undefined
            }
            ArrayStorage::Sparse { storage, length } => {
                let value = storage.get(&0).copied().unwrap_or(Value::Undefined);
                storage.remove(&0);

                let mut new_storage = BTreeMap::new();
                for (&key, &value) in storage.iter() {
                    new_storage.insert(key - 1, value);
                }

                *storage = new_storage;

                if *length > 0 {
                    *length -= 1;
                }
                self.maybe_convert_to_dense();
                value
            }
        }
    }

    /// Unshift a single value onto the start of this array.
    ///
    /// It is not possible to push a hole onto the array.
    pub fn unshift(&mut self, item: Value<'gc>) {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                storage.insert(0, Some(item));
                *occupied_count += 1;
            }
            ArrayStorage::Sparse { storage, length } => {
                let mut new_storage = BTreeMap::new();
                new_storage.insert(0, item);
                for (key, value) in storage.iter() {
                    new_storage.insert(key + 1, *value);
                }
                *storage = new_storage;
                *length += 1;
            }
        }
    }

    /// Iterate over array values.
    pub fn iter<'a>(
        &'a self,
    ) -> impl DoubleEndedIterator<Item = Option<Value<'gc>>>
    + ExactSizeIterator<Item = Option<Value<'gc>>>
    + 'a {
        let index_back = self.length();
        ArrayStorageIterator {
            storage: self,
            index: 0,
            index_back,
        }
    }

    /// Remove a value from a specific position in the array.
    ///
    /// This function returns None if the index is out of bonds.
    /// Otherwise, it returns the removed value.
    ///
    /// Negative bounds are supported and treated as indexing from the end of
    /// the array, backwards.
    pub fn remove(&mut self, position: i32) -> Option<Value<'gc>> {
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                let position = if position < 0 {
                    std::cmp::max(position + storage.len() as i32, 0) as usize
                } else {
                    position as usize
                };

                if position >= storage.len() {
                    None
                } else {
                    let value = storage.remove(position);
                    if value.is_some() {
                        *occupied_count -= 1;
                    }
                    self.maybe_convert_to_sparse();
                    value
                }
            }
            ArrayStorage::Sparse { storage, length } => {
                let position = if position < 0 {
                    std::cmp::max(position + *length as i32, 0) as usize
                } else {
                    position as usize
                };

                if position >= *length {
                    None
                } else {
                    let value = storage.get(&position).copied();
                    storage.remove(&position);
                    *length -= 1;
                    let new_storage = storage.split_off(&position);
                    for (&key, &value) in new_storage.iter() {
                        storage.insert(key - 1, value);
                    }
                    self.maybe_convert_to_dense();
                    value
                }
            }
        }
    }

    pub fn splice(
        &mut self,
        start: usize,
        delete_count: usize,
        items: &[Value<'gc>],
    ) -> ArrayStorage<'gc> {
        let delete_count = delete_count.min(self.length() - start);
        let end = start + delete_count;
        match self {
            ArrayStorage::Dense {
                storage,
                occupied_count,
            } => {
                let occupied_added = items.len();
                let mut occupied_removed = 0;
                let splice = storage
                    .splice(start..end, items.iter().map(|i| Some(*i)))
                    .inspect(|v| {
                        if v.is_some() {
                            occupied_removed += 1;
                        }
                    })
                    .collect::<Vec<_>>();
                *occupied_count += occupied_added;
                *occupied_count -= occupied_removed;
                ArrayStorage::from_storage(splice)
            }
            ArrayStorage::Sparse { storage, length } => {
                // 1. Split the map into 3 ranges:
                //      storage, splice, remaining
                let remaining = storage.split_off(&end);
                let splice = storage.split_off(&start);

                // 2. Remap indices in remaining elements
                let rshift = items.len() as isize - delete_count as isize;
                let mut remaining = if rshift == 0 {
                    remaining
                } else {
                    remaining
                        .into_iter()
                        .map(|(i, v)| (i.saturating_add_signed(rshift), v))
                        .collect::<BTreeMap<_, _>>()
                };

                // 3. Put remaining elements back in original array
                storage.append(&mut remaining);

                // 4. Put new items to the original array
                for (i, item) in items.iter().enumerate() {
                    storage.insert(start + i, *item);
                }

                // 5. Remap indices in splice
                let splice = if start == 0 {
                    splice
                } else {
                    splice
                        .into_iter()
                        .map(|(i, v)| (i.saturating_sub(start), v))
                        .collect::<BTreeMap<_, _>>()
                };

                // 6. Update length
                *length = length.saturating_add_signed(rshift);

                ArrayStorage::Sparse {
                    storage: splice,
                    length: delete_count,
                }
            }
        }
    }
}

impl<'gc, V> FromIterator<V> for ArrayStorage<'gc>
where
    V: Into<Value<'gc>>,
{
    fn from_iter<I>(values: I) -> Self
    where
        I: IntoIterator<Item = V>,
    {
        let storage: Vec<_> = values.into_iter().map(|v| Some(v.into())).collect();
        ArrayStorage::Dense {
            occupied_count: storage.len(),
            storage,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    macro_rules! assert_occupied_count {
        ($storage:expr, $count:literal) => {{
            if let ArrayStorage::Dense {
                ref storage,
                occupied_count,
            } = $storage
            {
                assert_eq!(occupied_count, $count);
                let real_occupied = storage.iter().filter(|v| v.is_some()).count();
                assert_eq!(real_occupied, $count);
            } else {
                panic!("Expected ArrayStorage to be dense");
            }
        }};
    }

    #[test]
    fn test_occupied_count_from_iter() {
        let storage = ArrayStorage::from_iter([1, 2, 3, 4, 5]);
        assert_occupied_count!(storage, 5);

        let storage = ArrayStorage::from_iter::<[i32; 0]>([]);
        assert_occupied_count!(storage, 0);

        let storage = ArrayStorage::from_iter([Value::Null, Value::Undefined]);
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_new() {
        let storage = ArrayStorage::new(0);
        assert_occupied_count!(storage, 0);

        let storage = ArrayStorage::new(42);
        assert_occupied_count!(storage, 0);
    }

    #[test]
    fn test_occupied_count_from_storage() {
        let storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);
    }

    #[test]
    fn test_occupied_count_set_no_extend() {
        let mut storage = ArrayStorage::new(3);
        assert_occupied_count!(storage, 0);

        storage.set(0, Value::Null);
        assert_occupied_count!(storage, 1);

        storage.set(0, Value::Undefined);
        assert_occupied_count!(storage, 1);

        storage.set(2, Value::Integer(5));
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_set_extend() {
        let mut storage = ArrayStorage::new(3);
        assert_occupied_count!(storage, 0);

        storage.set(17, Value::Null);
        assert_occupied_count!(storage, 1);

        storage.set(25, Value::Undefined);
        assert_occupied_count!(storage, 2);

        storage.set(25, Value::Null);
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_delete() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);

        storage.delete(2);
        assert_occupied_count!(storage, 3);

        storage.delete(0);
        assert_occupied_count!(storage, 2);

        storage.delete(15);
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_set_length_extend() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);

        storage.set_length(10);
        assert_occupied_count!(storage, 3);
    }

    #[test]
    fn test_occupied_count_set_length_shrink_holes() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);

        storage.set_length(5);
        assert_occupied_count!(storage, 3);

        storage.set_length(4);
        assert_occupied_count!(storage, 3);
    }

    #[test]
    fn test_occupied_count_set_length_shrink_occupied() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);

        storage.set_length(3);
        assert_occupied_count!(storage, 2);

        storage.set_length(2);
        assert_occupied_count!(storage, 2);

        storage.set_length(1);
        assert_occupied_count!(storage, 1);

        storage.set_length(0);
        assert_occupied_count!(storage, 0);
    }

    #[test]
    fn test_occupied_count_set_length_shrink_faster_to_count_deleted() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
            Some(Value::Integer(5)),
            None,
            None,
            None,
            Some(Value::Integer(5)),
            None,
            None,
            Some(Value::Integer(5)),
            None,
            None,
            Some(Value::Integer(5)),
            None,
        ]);
        assert_occupied_count!(storage, 7);

        storage.set_length(12);
        assert_occupied_count!(storage, 5);
    }

    #[test]
    fn test_occupied_count_set_length_shrink_faster_to_count_remaining() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
            Some(Value::Integer(5)),
            None,
            None,
            None,
            Some(Value::Integer(5)),
            None,
            None,
            Some(Value::Integer(5)),
            None,
            None,
            Some(Value::Integer(5)),
            None,
        ]);
        assert_occupied_count!(storage, 7);

        storage.set_length(5);
        assert_occupied_count!(storage, 3);
    }

    #[test]
    fn test_occupied_count_push() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);

        storage.push(Value::Null);
        assert_occupied_count!(storage, 4);
    }

    #[test]
    fn test_occupied_count_push_hole() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Integer(5)),
            None,
            None,
        ]);
        assert_occupied_count!(storage, 3);

        storage.push_hole();
        assert_occupied_count!(storage, 3);
    }

    #[test]
    fn test_occupied_count_pop_hole() {
        let mut storage =
            ArrayStorage::from_storage(vec![Some(Value::Null), Some(Value::Undefined), None]);
        assert_occupied_count!(storage, 2);

        storage.pop();
        assert_occupied_count!(storage, 1);

        storage.pop();
        assert_occupied_count!(storage, 0);
    }

    #[test]
    fn test_occupied_count_pop_non_hole() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Undefined),
        ]);
        assert_occupied_count!(storage, 3);

        storage.pop();
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_pop_all_holes() {
        let mut storage = ArrayStorage::new(10);
        assert_occupied_count!(storage, 0);

        storage.pop();
        assert_occupied_count!(storage, 0);
    }

    #[test]
    fn test_occupied_count_shift_hole() {
        let mut storage =
            ArrayStorage::from_storage(vec![None, Some(Value::Null), Some(Value::Undefined)]);
        assert_occupied_count!(storage, 2);

        storage.shift();
        assert_occupied_count!(storage, 2);

        storage.shift();
        assert_occupied_count!(storage, 1);
    }

    #[test]
    fn test_occupied_count_shift_non_hole() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Undefined),
            None,
            Some(Value::Undefined),
        ]);
        assert_occupied_count!(storage, 3);

        storage.shift();
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_shift_all_holes() {
        let mut storage = ArrayStorage::new(10);
        assert_occupied_count!(storage, 0);

        storage.shift();
        assert_occupied_count!(storage, 0);
    }

    #[test]
    fn test_occupied_count_unshift_empty() {
        let mut storage = ArrayStorage::new(0);
        assert_occupied_count!(storage, 0);

        storage.unshift(Value::Null);
        assert_occupied_count!(storage, 1);
    }

    #[test]
    fn test_occupied_count_unshift_holes() {
        let mut storage = ArrayStorage::new(5);
        assert_occupied_count!(storage, 0);

        storage.unshift(Value::Null);
        assert_occupied_count!(storage, 1);
    }

    #[test]
    fn test_occupied_count_unshift_non_holes() {
        let mut storage =
            ArrayStorage::from_storage(vec![Some(Value::Null), None, Some(Value::Undefined)]);
        assert_occupied_count!(storage, 2);

        storage.unshift(Value::Null);
        assert_occupied_count!(storage, 3);
    }

    #[test]
    fn test_occupied_count_remove_hole() {
        let mut storage =
            ArrayStorage::from_storage(vec![Some(Value::Null), None, Some(Value::Undefined)]);
        assert_occupied_count!(storage, 2);

        storage.remove(1);
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_remove_non_hole() {
        let mut storage =
            ArrayStorage::from_storage(vec![Some(Value::Null), None, Some(Value::Undefined)]);
        assert_occupied_count!(storage, 2);

        storage.remove(2);
        assert_occupied_count!(storage, 1);
    }

    #[test]
    fn test_occupied_count_remove_oob() {
        let mut storage =
            ArrayStorage::from_storage(vec![Some(Value::Null), None, Some(Value::Undefined)]);
        assert_occupied_count!(storage, 2);

        storage.remove(5);
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_splice_delete() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Integer(5)),
            None,
            Some(Value::Undefined),
        ]);
        assert_occupied_count!(storage, 3);

        storage.splice(1, 2, &[]);
        assert_occupied_count!(storage, 2);
    }

    #[test]
    fn test_occupied_count_splice_insert() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Integer(5)),
            None,
            Some(Value::Undefined),
        ]);
        assert_occupied_count!(storage, 3);

        storage.splice(1, 0, &[Value::Null, Value::Integer(0)]);
        assert_occupied_count!(storage, 5);
    }

    #[test]
    fn test_occupied_count_splice_insert_and_delete() {
        let mut storage = ArrayStorage::from_storage(vec![
            Some(Value::Null),
            Some(Value::Integer(5)),
            None,
            Some(Value::Undefined),
        ]);
        assert_occupied_count!(storage, 3);

        storage.splice(1, 2, &[Value::Null, Value::Integer(0)]);
        assert_occupied_count!(storage, 4);
    }
}
