# This code is part of Qiskit.
#
# (C) Copyright IBM 2024.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""Generic BackendV2 class that with a simulated ``run``."""

from __future__ import annotations
import warnings

import numpy as np

from qiskit.circuit import QuantumCircuit, Instruction
from qiskit.circuit.controlflow import (
    BoxOp,
    IfElseOp,
    WhileLoopOp,
    ForLoopOp,
    SwitchCaseOp,
    BreakLoopOp,
    ContinueLoopOp,
)
from qiskit.circuit.library.standard_gates import get_standard_gate_name_mapping
from qiskit.exceptions import QiskitError
from qiskit.transpiler import CouplingMap, Target, InstructionProperties, QubitProperties
from qiskit.providers import Options
from qiskit.providers.basic_provider import BasicSimulator
from qiskit.providers.backend import BackendV2
from qiskit.utils import optionals as _optionals

# Noise default values/ranges for duration and error of supported
# instructions. There are two possible formats:
# - (min_duration, max_duration, min_error, max_error),
#   if the defaults are ranges.
# - (duration, error), if the defaults are fixed values.
_NOISE_DEFAULTS = {
    "cx": (7.992e-08, 8.99988e-07, 1e-5, 5e-3),
    "ecr": (7.992e-08, 8.99988e-07, 1e-5, 5e-3),
    "cz": (7.992e-08, 8.99988e-07, 1e-5, 5e-3),
    "id": (2.997e-08, 5.994e-08, 9e-5, 1e-4),
    "rz": (0.0, 0.0),
    "sx": (2.997e-08, 5.994e-08, 9e-5, 1e-4),
    "x": (2.997e-08, 5.994e-08, 9e-5, 1e-4),
    "measure": (6.99966e-07, 1.500054e-06, 1e-5, 5e-3),
    "delay": (None, None),
    "reset": (None, None),
}

# Fallback values for gates with unknown noise default ranges.
_NOISE_DEFAULTS_FALLBACK = {
    "1-q": (2.997e-08, 5.994e-08, 9e-5, 1e-4),
    "multi-q": (7.992e-08, 8.99988e-07, 5e-3),
}

# Ranges to sample qubit properties from.
_QUBIT_PROPERTIES = {
    "dt": 0.222e-9,
    "t1": (100e-6, 200e-6),
    "t2": (100e-6, 200e-6),
    "frequency": (5e9, 5.5e9),
}


class GenericBackendV2(BackendV2):
    """Generic :class:`~.BackendV2` implementation with a configurable constructor. This class will
    return a :class:`~.BackendV2` instance that runs on a local simulator (in the spirit of fake
    backends) and contains all the necessary information to test backend-interfacing components, such
    as the transpiler. A :class:`.GenericBackendV2` instance can be constructed from as little as a
    specified ``num_qubits``, but users can additionally configure the basis gates, coupling map,
    ability to run dynamic circuits (control flow instructions) and dtm.
    The remainder of the backend properties are generated by randomly sampling
    from default ranges extracted from historical IBM backend data. The seed for this random
    generation can be fixed to ensure the reproducibility of the backend output.
    This backend only supports gates in the standard library, if you need a more flexible backend,
    there is always the option to directly instantiate a :class:`.Target` object to use for
    transpilation.
    """

    def __init__(
        self,
        num_qubits: int,
        basis_gates: list[str] | None = None,
        *,
        coupling_map: list[list[int]] | CouplingMap | None = None,
        control_flow: bool = False,
        dtm: float | None = None,
        dt: float | None = None,
        seed: int | None = None,
        noise_info: bool = True,
    ):
        """
        Args:
            num_qubits: Number of qubits that will be used to construct the backend's target.
                Note that, while there is no limit in the size of the target that can be
                constructed, this backend runs on local noisy simulators, and these might
                present limitations in the number of qubits that can be simulated.

            basis_gates: List of basis gate names to be supported by
                the target. These must be part of the standard qiskit circuit library.
                The default set of basis gates is ``["id", "rz", "sx", "x", "cx"]``
                The ``"reset"``,  ``"delay"``, and ``"measure"`` instructions are
                always supported by default, even if not specified via ``basis_gates``.

            coupling_map: Optional coupling map
                for the backend. Multiple formats are supported:

                #. :class:`~.CouplingMap` instance
                #. List, must be given as an edge list representing the two qubit interactions
                   supported by the backend, for example:
                   ``[[0, 1], [0, 3], [1, 2], [1, 5], [2, 5], [4, 1], [5, 3]]``

                If ``coupling_map`` is specified, it must match the number of qubits
                specified in ``num_qubits``. If ``coupling_map`` is not specified,
                a fully connected coupling map will be generated with ``num_qubits``
                qubits.

            control_flow: Flag to enable control flow directives on the target
                (defaults to False).

            dtm: System time resolution of output signals in nanoseconds.
                None by default.

            dt: System time resolution of input signals in nanoseconds.
                None by default.

            seed: Optional seed for generation of default values.

            noise_info: If true, associates gates and qubits with default noise information.
        """

        super().__init__(
            provider=None,
            name=f"generic_backend_{num_qubits}q",
            description=f"This is a device with {num_qubits} qubits and generic settings.",
            backend_version="",
        )

        self._sim = None
        self._rng = np.random.default_rng(seed=seed)
        self._dtm = dtm
        self._dt = dt
        self._num_qubits = num_qubits
        self._control_flow = control_flow
        self._supported_gates = get_standard_gate_name_mapping()
        self._noise_info = noise_info

        if coupling_map is None:
            self._coupling_map = CouplingMap().from_full(num_qubits)
        else:
            if isinstance(coupling_map, CouplingMap):
                self._coupling_map = coupling_map
            else:
                self._coupling_map = CouplingMap(coupling_map)

            if num_qubits != self._coupling_map.size():
                raise QiskitError(
                    f"The number of qubits (got {num_qubits}) must match "
                    f"the size of the provided coupling map (got {self._coupling_map.size()})."
                )

        self._basis_gates = (
            basis_gates if basis_gates is not None else ["cx", "id", "rz", "sx", "x"]
        )
        for name in ["reset", "delay", "measure"]:
            if name not in self._basis_gates:
                self._basis_gates.append(name)

        self._build_generic_target()

    @property
    def target(self):
        return self._target

    @property
    def max_circuits(self):
        return None

    @property
    def dtm(self) -> float:
        """Return the system time resolution of output signals"""
        # converting `dtm` from nanoseconds to seconds
        return self._dtm * 1e-9 if self._dtm is not None else None

    @property
    def meas_map(self) -> list[list[int]]:
        return self._target.concurrent_measurements

    def _get_noise_defaults(self, name: str, num_qubits: int) -> tuple:
        """Return noise default values/ranges for duration and error of supported
        instructions. There are two possible formats:
            - (min_duration, max_duration, min_error, max_error),
              if the defaults are ranges.
            - (duration, error), if the defaults are fixed values.
        """
        if name in _NOISE_DEFAULTS:
            return _NOISE_DEFAULTS[name]
        if num_qubits == 1:
            return _NOISE_DEFAULTS_FALLBACK["1-q"]
        return _NOISE_DEFAULTS_FALLBACK["multi-q"]

    def _build_generic_target(self):
        """This method generates a :class:`~.Target` instance with
        default qubit and instruction properties.
        """
        # the qubit properties are sampled from default ranges
        properties = _QUBIT_PROPERTIES
        if not self._noise_info:
            self._target = Target(
                description=f"Generic Target with {self._num_qubits} qubits",
                num_qubits=self._num_qubits,
                dt=properties["dt"] if self._dt is None else self._dt,
                qubit_properties=None,
                concurrent_measurements=[list(range(self._num_qubits))],
            )
        else:
            self._target = Target(
                description=f"Generic Target with {self._num_qubits} qubits",
                num_qubits=self._num_qubits,
                dt=properties["dt"] if self._dt is None else self._dt,
                qubit_properties=[
                    QubitProperties(
                        t1=self._rng.uniform(properties["t1"][0], properties["t1"][1]),
                        t2=self._rng.uniform(properties["t2"][0], properties["t2"][1]),
                        frequency=self._rng.uniform(
                            properties["frequency"][0], properties["frequency"][1]
                        ),
                    )
                    for _ in range(self._num_qubits)
                ],
                concurrent_measurements=[list(range(self._num_qubits))],
            )

        # Iterate over gates, generate noise params from defaults,
        # and add instructions and noise information to the target.
        for name in self._basis_gates:
            if name not in self._supported_gates:
                raise QiskitError(
                    f"Provided basis gate {name} is not an instruction "
                    f"in the standard qiskit circuit library."
                )
            gate = self._supported_gates[name]
            if self.num_qubits < gate.num_qubits:
                raise QiskitError(
                    f"Provided basis gate {name} needs more qubits than {self.num_qubits}, "
                    f"which is the size of the backend."
                )
            if self._noise_info:
                noise_params = self._get_noise_defaults(name, gate.num_qubits)
                self._add_noisy_instruction_to_target(gate, noise_params)
            else:
                qarg_set = self._coupling_map if gate.num_qubits > 1 else range(self.num_qubits)
                props = {(qarg,) if isinstance(qarg, int) else qarg: None for qarg in qarg_set}
                self._target.add_instruction(gate, properties=props, name=name)

        if self._control_flow:
            self._target.add_instruction(IfElseOp, name="if_else")
            self._target.add_instruction(WhileLoopOp, name="while_loop")
            self._target.add_instruction(ForLoopOp, name="for_loop")
            self._target.add_instruction(SwitchCaseOp, name="switch_case")
            self._target.add_instruction(BreakLoopOp, name="break")
            self._target.add_instruction(ContinueLoopOp, name="continue")
            self._target.add_instruction(BoxOp, name="box")

    def _add_noisy_instruction_to_target(
        self,
        instruction: Instruction,
        noise_params: tuple[float, ...] | None,
    ) -> None:
        """Add instruction properties to target for specified instruction.

        Args:
            instruction: Instance of instruction to be added to the target
            noise_params: Error and duration noise values/ranges to
                include in instruction properties.
        """
        qarg_set = self._coupling_map if instruction.num_qubits > 1 else range(self.num_qubits)
        props = {}
        for qarg in qarg_set:
            try:
                qargs = tuple(qarg)
            except TypeError:
                qargs = (qarg,)
            duration, error = (
                noise_params
                if len(noise_params) == 2
                else (
                    self._rng.uniform(*noise_params[:2]),
                    self._rng.uniform(*noise_params[2:]),
                )
            )

            if duration is not None and len(noise_params) > 2:
                # Ensure exact conversion of duration from seconds to dt
                dt = _QUBIT_PROPERTIES["dt"]
                rounded_duration = round(duration / dt) * dt
                # Clamp rounded duration to be between min and max values
                duration = max(noise_params[0], min(rounded_duration, noise_params[1]))
            props.update({qargs: InstructionProperties(duration, error)})

        self._target.add_instruction(instruction, props)

    def run(self, run_input, **options):
        """Run on the backend using a simulator.

        This method runs circuit jobs (an individual or a list of :class:`~.QuantumCircuit`
        ) using :class:`~.BasicSimulator` or Aer simulator and returns a
        :class:`~qiskit.providers.Job` object.

        If qiskit-aer is installed, jobs will be run using the ``AerSimulator`` with
        noise model of the backend. Otherwise, jobs will be run using the
        ``BasicSimulator`` simulator without noise.

        Args:
            run_input (QuantumCircuit or list): An
                individual or a list of :class:`~qiskit.circuit.QuantumCircuit`
                objects to run on the backend.
            options: Any kwarg options to pass to the backend for running the
                config. If a key is also present in the options
                attribute/object, then the expectation is that the value
                specified will be used instead of what's set in the options
                object.

        Returns:
            Job: The job object for the run

        Raises:
            QiskitError: If input is not :class:`~qiskit.circuit.QuantumCircuit` or a list of
            :class:`~qiskit.circuit.QuantumCircuit` objects.
        """
        circuits = run_input
        if not isinstance(circuits, QuantumCircuit) and (
            not isinstance(circuits, list)
            or not all(isinstance(x, QuantumCircuit) for x in circuits)
        ):
            raise QiskitError(
                f"Invalid input object {circuits}, must be either a "
                "QuantumCircuit or a list of QuantumCircuit objects"
            )

        if not _optionals.HAS_AER:
            warnings.warn("Aer not found using BasicSimulator and no noise", RuntimeWarning)
        if self._sim is None:
            self._setup_sim()
        self._sim._options = self._options
        job = self._sim.run(circuits, **options)
        return job

    def _setup_sim(self) -> None:
        if _optionals.HAS_AER:
            from qiskit_aer import AerSimulator
            from qiskit_aer.noise import NoiseModel

            self._sim = AerSimulator()
            noise_model = NoiseModel.from_backend(self)
            self._sim.set_options(noise_model=noise_model)
            # Update backend default too to avoid overwriting
            # it when run() is called
            self.set_options(noise_model=noise_model)
        else:
            self._sim = BasicSimulator()

    @classmethod
    def _default_options(cls) -> Options:
        if _optionals.HAS_AER:
            from qiskit_aer import AerSimulator

            return AerSimulator._default_options()
        else:
            return BasicSimulator._default_options()
