# frozen_string_literal: true

# Licensed to the Software Freedom Conservancy (SFC) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

module Selenium
  module WebDriver
    module SpecSupport
      module Helpers
        def driver
          GlobalTestEnv.driver_instance
        end

        def reset_driver!(...)
          GlobalTestEnv.reset_driver!(...)
        end

        def quit_driver
          GlobalTestEnv.quit_driver
        end

        def create_driver!(...)
          GlobalTestEnv.create_driver!(...)
        end

        def url_for(filename)
          GlobalTestEnv.url_for filename
        end

        def fix_windows_path(path)
          return path unless WebDriver::Platform.windows?

          if GlobalTestEnv.browser == :ie
            path = path[%r{file://(.*)}, 1]
            path = WebDriver::Platform.windows_path(path)

            "file://#{path}"
          else
            path.sub(%r[file:/{0,2}], 'file:///')
          end
        end

        def long_wait
          @long_wait ||= Wait.new(timeout: 30)

          return @long_wait unless block_given?

          result = nil
          @long_wait.until { result = yield }
          result
        end

        def short_wait
          @short_wait ||= Wait.new(timeout: 3)

          return @short_wait unless block_given?

          result = nil
          @short_wait.until { result = yield }
          result
        end

        def wait_for_alert
          wait = Wait.new(timeout: 5, ignore: Error::NoSuchAlertError)
          wait.until { driver.switch_to.alert }
        end

        def wait_for_no_alert
          wait = Wait.new(timeout: 5, ignore: Error::UnexpectedAlertOpenError)
          wait.until { driver.title }
        end

        def wait_for_element(locator, timeout = 25)
          wait = Wait.new(timeout: timeout, ignore: Error::NoSuchElementError, message_provider: lambda {
            url = "page url: #{driver.current_url};\n"
            source = "page source: #{driver.find_element(css: 'body').attribute('innerHTML')}\n"
            "could not find element #{locator} in #{timeout} seconds;\n#{url}#{source}"
          })
          wait.until { driver.find_element(locator) }
        end

        def wait_for_url(new_url, timeout = 15)
          wait = Wait.new(timeout: timeout, message_provider: lambda {
            "could not wait for URL #{new_url} in #{timeout} seconds;\nactual page url: #{driver.current_url};\n"
          })
          wait.until do
            driver.current_url.include?(new_url)
          end
        end

        def wait_for_devtools_target(target_type:)
          wait = Wait.new(timeout: 3, ignore: Error::NoSuchTargetError)
          wait.until { driver.devtools(target_type: target_type).target }
        end

        def wait_for_title(title:)
          wait = Wait.new(timeout: 15)
          wait.until { driver.title == title }
        end

        def open_file(file_name)
          driver.navigate.to 'about:blank'
          driver.navigate.to url_for('blank.html')
          driver.navigate.to url_for(file_name)
          wait_for_url(file_name)
        end

        def wait(timeout = 10)
          Wait.new(timeout: timeout)
        end

        def png_size(path)
          png = File.read(path, mode: 'rb')[0x10..0x18]
          width = png.unpack1('NN')
          height = png.unpack('NN').last

          if Platform.mac? # Retina
            width /= 2
            height /= 2
          end

          [width, height]
        end

        def create_tempfile
          Tempfile.new.tap do |file|
            file.write('This is a dummy test file')
            file.close
          end
        end
      end # Helpers
    end # SpecSupport
  end # WebDriver
end # Selenium
