/* SPDX-License-Identifier: LGPL-2.1+ */
#include"util.h"
#include "module_configure.h"
#include <dirent.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/wait.h>
#include <openssl/opensslv.h>

#if OPENSSL_VERSION_NUMBER < 0x30000000L
#define USE_OPENSSL_1_1
#include <openssl/md5.h>
#include <openssl/sha.h>
#else
#define USE_OPENSSL_3_0
#include <openssl/evp.h>
#endif


#define MAX_ALLOWED_SHELL_MD5S_NUM 128

void freep(void *p) {
    if (*(void**)p)
        free(*(void**) p);
}

static inline void *mfree(void *memory) {
        free(memory);
        return NULL;
}

void strv_clear(char **l) {
        char **k;

        if (!l)
                return;

        for (k = l; *k; k++)
                free(*k);

        *l = NULL;
}

char **strv_free(char **l) {
        strv_clear(l);
        return mfree(l);
}

int str_endsWith(const char *str, const char *suffix) {
    if (!str || !suffix) return 0;

    size_t str_len = strlen(str);
    size_t suffix_len = strlen(suffix);

    if (suffix_len > str_len) return 0;

    return strcmp(str + str_len - suffix_len, suffix) == 0;
}


#ifdef USE_OPENSSL_1_1
int calculateFileMD5_openssl1_1(const char* file_path, unsigned char* md5Digest) {
    FILE* file = fopen(file_path, "rb");
    if (!file) {
        perror("Error opening file");
        return -1;
    }

    MD5_CTX md5Context;
    MD5_Init(&md5Context);

    unsigned char buffer[1024];
    int bytesRead = 0;

    while ((bytesRead = fread(buffer, 1, sizeof(buffer), file)) > 0) {
        MD5_Update(&md5Context, buffer, bytesRead);
    }

    if (ferror(file)) {
        perror("Error reading file");
        fclose(file);
        return -1;
    }

    fclose(file);
    MD5_Final(md5Digest, &md5Context);

    return 0;
}

int calculate_sha256_openssl1_1(const char* file_path, unsigned char* sha256_hash) {
    FILE* file = fopen(file_path, "rb");
    if (file == NULL) {
        fprintf(stderr, "Failed to open file: %s\n", file_path);
        return -1;
    }

    SHA256_CTX ctx;
    SHA256_Init(&ctx);

    unsigned char buffer[4096];
    size_t bytes_read;

    while ((bytes_read = fread(buffer, 1, sizeof(buffer), file)) > 0) {
        SHA256_Update(&ctx, buffer, bytes_read);
    }

    SHA256_Final(sha256_hash, &ctx);
    if (ferror(file)) {
        perror("Error reading file");
        fclose(file);
        return -1;
    }

    fclose(file);
    return 0;
}
#endif

#ifdef USE_OPENSSL_3_0

int calculateFileMD5_openssl3_0(const char* file_path, unsigned char* md5Digest) {
    FILE* file = fopen(file_path, "rb");
    if (!file) {
        perror("Error opening file");
        return -1;
    }

    EVP_MD_CTX *md_ctx = EVP_MD_CTX_new();
    const EVP_MD *md = EVP_md5();

    if (md_ctx == NULL || md == NULL) {
        fprintf(stderr, "Failed to create MD5 context\n");
        fclose(file);
        return -1;
    }

    if (EVP_DigestInit_ex(md_ctx, md, NULL) <= 0) {
        fprintf(stderr, "Failed to initialize MD5 context\n");
        EVP_MD_CTX_free(md_ctx);
        fclose(file);
        return -1;
    }

    unsigned char buffer[1024];
    size_t bytesRead;

    while ((bytesRead = fread(buffer, 1, sizeof(buffer), file)) > 0) {
        if (EVP_DigestUpdate(md_ctx, buffer, bytesRead) <= 0) {
            fprintf(stderr, "Failed to update MD5 digest\n");
            EVP_MD_CTX_free(md_ctx);
            fclose(file);
            return -1;
        }
    }

    if (ferror(file)) {
        perror("Error reading file");
        EVP_MD_CTX_free(md_ctx);
        fclose(file);
        return -1;
    }

    if (EVP_DigestFinal_ex(md_ctx, md5Digest, NULL) <= 0) {
        fprintf(stderr, "Failed to finalize MD5 digest\n");
        EVP_MD_CTX_free(md_ctx);
        fclose(file);
        return -1;
    }

    EVP_MD_CTX_free(md_ctx);
    fclose(file);

    return 0;
}

int calculate_sha256_openssl3_0(const char* file_path, unsigned char* sha256_hash) {
    FILE* file = fopen(file_path, "rb");
    if (file == NULL) {
        fprintf(stderr, "Failed to open file: %s\n", file_path);
        return -1;
    }

    EVP_MD_CTX *md_ctx = EVP_MD_CTX_new();
    const EVP_MD *md = EVP_sha256();

    if (md_ctx == NULL || md == NULL) {
        fprintf(stderr, "Failed to create SHA-256 context\n");
        fclose(file);
        return -1;
    }

    if (EVP_DigestInit_ex(md_ctx, md, NULL) <= 0) {
        fprintf(stderr, "Failed to initialize SHA-256 context\n");
        EVP_MD_CTX_free(md_ctx);
        fclose(file);
        return -1;
    }

    unsigned char buffer[4096];
    size_t bytes_read;

    while ((bytes_read = fread(buffer, 1, sizeof(buffer), file)) > 0) {
        if (EVP_DigestUpdate(md_ctx, buffer, bytes_read) <= 0) {
            fprintf(stderr, "Failed to update SHA-256 digest\n");
            EVP_MD_CTX_free(md_ctx);
            fclose(file);
            return -1;
        }
    }

    if (ferror(file)) {
        perror("Error reading file");
        EVP_MD_CTX_free(md_ctx);
        fclose(file);
        return -1;
    }

    if (EVP_DigestFinal_ex(md_ctx, sha256_hash, NULL) <= 0) {
        fprintf(stderr, "Failed to finalize SHA-256 digest\n");
        EVP_MD_CTX_free(md_ctx);
        fclose(file);
        return -1;
    }

    EVP_MD_CTX_free(md_ctx);
    fclose(file);

    return 0;
}
#endif // USE_OPENSSL_3_0

int calculateFileMD5(const char* file_path, unsigned char* md5Digest) {
#ifdef USE_OPENSSL_1_1
    return calculateFileMD5_openssl1_1(file_path, md5Digest);
#elif defined(USE_OPENSSL_3_0)
    return calculateFileMD5_openssl3_0(file_path, md5Digest);
#else
    #error "Unsupported OpenSSL version"
#endif
}

int calculate_sha256(const char* file_path, unsigned char* sha256_hash) {
#ifdef USE_OPENSSL_1_1
    return calculate_sha256_openssl1_1(file_path, sha256_hash);
#elif defined(USE_OPENSSL_3_0)
    return calculate_sha256_openssl3_0(file_path, sha256_hash);
#else
    #error "Unsupported OpenSSL version"
#endif
}

// Read data from file descriptor and store it in output
int read_from_fd(int fd, char **output) {
    if (fd < 0 || !output) return ERROR;

    FILE *fp = fdopen(fd, "r");
    if (!fp) return ERROR;

    char buffer[4096];
    size_t total_size = 0;
    size_t all_size = 4096;
    *output = malloc(all_size);
    if (!*output) {
        fclose(fp);
        return ERROR;
    }

    while (fgets(buffer, sizeof(buffer), fp)) {
        size_t chunk_len = strlen(buffer);
        if (total_size + chunk_len + 1 > all_size) {
            all_size = total_size + chunk_len*2 + 1;
            char *new_output = realloc(*output, all_size);
            if (!new_output) {
                free(*output);
                *output = NULL;
                fclose(fp);
                return ERROR;
            }
            *output = new_output;
        }
        memcpy(*output + total_size, buffer, chunk_len);
        total_size += chunk_len;
        (*output)[total_size] = '\0';
    }

    fclose(fp);
    return OK;
}

int start_process(const char *cmd_path, const char *arg_string, char **output) {
    if (!cmd_path || !arg_string)
        return ERROR;
    // Check if arg_string contains dangerous characters like semicolon, etc.
    if (strchr(arg_string, ';') != NULL || strchr(arg_string, '|') != NULL ||
        strchr(arg_string, '&') != NULL || strchr(arg_string, '>') != NULL ||
        strchr(arg_string, '<') != NULL) {
        fprintf(stderr, "Error: The argument string cannot contain special characters (;|&><).\n");
        return ERROR;
    }

    // Assuming the parameters are space-separated
    // Count how many spaces are in the string to allocate memory for the argument array
    int num_args = 1;  // At least one argument, which is the name itself
    for (int i = 0; arg_string[i] != '\0'; i++) {
        if (arg_string[i] == ' ') {
            num_args++;
        }
    }

    // Create an argument array of size num_args + 1 (the last NULL is for execvp)
    char **args = malloc((num_args + 2) * sizeof(char *)); // +2 for the path and NULL

    // The first argument is the path (e.g., "/xxx/xxx")
    args[0] = (char *)cmd_path;

    // Use strtok to split the name string into tokens based on space
    char *name_copy = strdup(arg_string);  // Duplicate the arg_string string for safe splitting
    char *token = strtok(name_copy, " ");
    int i = 1;
    while (token != NULL) {
        args[i] = token;
        token = strtok(NULL, " ");
        i++;
    }

    args[i] = NULL;  // execvp needs the last argument to be NULL

    // Create a pipe
    int pipefd[2];
    if (pipe(pipefd) == -1) {
        perror("pipe failed");
        free(args);
        free(name_copy);
        return ERROR;
    }

    // Fork a child process
    pid_t pid = fork();
    if (pid == -1) {
        // Error handling: failed to create a child process
        perror("fork failed");
        close(pipefd[0]);
        close(pipefd[1]);
        free(args);
        free(name_copy);
        return ERROR;
    }

    if (pid == 0) {
        // Child process: close read end, redirect stdout to write end, execute command
        close(pipefd[0]);
        if (output && *output == NULL) {
            dup2(pipefd[1], STDOUT_FILENO);
        }
        close(pipefd[1]);
        if (execvp(args[0], args) == -1) {
            perror("execvp failed");
            exit(EXIT_FAILURE);
        }
        return ERROR;
    } else {
        // Parent process: close write end
        int exit_status = OK;
        close(pipefd[1]);
        if (output && *output == NULL) {
            // Read child process output
            if (read_from_fd(pipefd[0], output) != OK) {
                exit_status = ERROR;
            }
        }

        // In the parent process, wait for the child to finish
        int status;
        if (waitpid(pid, &status, 0) == -1) {
            perror("waitpid failed");
            exit_status = ERROR;
        }
         // Free allocated memory
        free(args);
        free(name_copy);
        close(pipefd[0]);
        if (exit_status != OK) return exit_status;

        // Check the exit status of the child process
        if (WIFEXITED(status)) {
            exit_status = WEXITSTATUS(status);
            if (exit_status != OK) {
                fprintf(stderr,"exec %s %s failed with exit status %d.\n", cmd_path,arg_string,exit_status);
            }
        } else if (WIFSIGNALED(status)) {
            exit_status = WTERMSIG(status);
            fprintf(stderr,"exec %s %s terminated by signal %d.\n", cmd_path,arg_string,exit_status);
        } else {
           fprintf(stderr,"exec %s %s terminated with unknown status.\n",cmd_path,arg_string);
        }

        return exit_status;
    }
}