﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/ServerSideEncryption.h>
#include <aws/s3-crt/model/SessionCredentials.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Xml {
class XmlDocument;
}  // namespace Xml
}  // namespace Utils
namespace S3Crt {
namespace Model {
class CreateSessionResult {
 public:
  AWS_S3CRT_API CreateSessionResult() = default;
  AWS_S3CRT_API CreateSessionResult(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);
  AWS_S3CRT_API CreateSessionResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);

  ///@{
  /**
   * <p>The server-side encryption algorithm used when you store objects in the
   * directory bucket.</p>  <p>When accessing data stored in Amazon FSx file
   * systems using S3 access points, the only valid server side encryption option is
   * <code>aws:fsx</code>.</p>
   */
  inline ServerSideEncryption GetServerSideEncryption() const { return m_serverSideEncryption; }
  inline void SetServerSideEncryption(ServerSideEncryption value) {
    m_serverSideEncryptionHasBeenSet = true;
    m_serverSideEncryption = value;
  }
  inline CreateSessionResult& WithServerSideEncryption(ServerSideEncryption value) {
    SetServerSideEncryption(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If you specify <code>x-amz-server-side-encryption</code> with
   * <code>aws:kms</code>, this header indicates the ID of the KMS symmetric
   * encryption customer managed key that was used for object encryption.</p>
   */
  inline const Aws::String& GetSSEKMSKeyId() const { return m_sSEKMSKeyId; }
  template <typename SSEKMSKeyIdT = Aws::String>
  void SetSSEKMSKeyId(SSEKMSKeyIdT&& value) {
    m_sSEKMSKeyIdHasBeenSet = true;
    m_sSEKMSKeyId = std::forward<SSEKMSKeyIdT>(value);
  }
  template <typename SSEKMSKeyIdT = Aws::String>
  CreateSessionResult& WithSSEKMSKeyId(SSEKMSKeyIdT&& value) {
    SetSSEKMSKeyId(std::forward<SSEKMSKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If present, indicates the Amazon Web Services KMS Encryption Context to use
   * for object encryption. The value of this header is a Base64 encoded string of a
   * UTF-8 encoded JSON, which contains the encryption context as key-value pairs.
   * This value is stored as object metadata and automatically gets passed on to
   * Amazon Web Services KMS for future <code>GetObject</code> operations on this
   * object.</p>
   */
  inline const Aws::String& GetSSEKMSEncryptionContext() const { return m_sSEKMSEncryptionContext; }
  template <typename SSEKMSEncryptionContextT = Aws::String>
  void SetSSEKMSEncryptionContext(SSEKMSEncryptionContextT&& value) {
    m_sSEKMSEncryptionContextHasBeenSet = true;
    m_sSEKMSEncryptionContext = std::forward<SSEKMSEncryptionContextT>(value);
  }
  template <typename SSEKMSEncryptionContextT = Aws::String>
  CreateSessionResult& WithSSEKMSEncryptionContext(SSEKMSEncryptionContextT&& value) {
    SetSSEKMSEncryptionContext(std::forward<SSEKMSEncryptionContextT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether to use an S3 Bucket Key for server-side encryption with KMS
   * keys (SSE-KMS).</p>
   */
  inline bool GetBucketKeyEnabled() const { return m_bucketKeyEnabled; }
  inline void SetBucketKeyEnabled(bool value) {
    m_bucketKeyEnabledHasBeenSet = true;
    m_bucketKeyEnabled = value;
  }
  inline CreateSessionResult& WithBucketKeyEnabled(bool value) {
    SetBucketKeyEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The established temporary security credentials for the created session.</p>
   */
  inline const SessionCredentials& GetCredentials() const { return m_credentials; }
  template <typename CredentialsT = SessionCredentials>
  void SetCredentials(CredentialsT&& value) {
    m_credentialsHasBeenSet = true;
    m_credentials = std::forward<CredentialsT>(value);
  }
  template <typename CredentialsT = SessionCredentials>
  CreateSessionResult& WithCredentials(CredentialsT&& value) {
    SetCredentials(std::forward<CredentialsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateSessionResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  ServerSideEncryption m_serverSideEncryption{ServerSideEncryption::NOT_SET};
  bool m_serverSideEncryptionHasBeenSet = false;

  Aws::String m_sSEKMSKeyId;
  bool m_sSEKMSKeyIdHasBeenSet = false;

  Aws::String m_sSEKMSEncryptionContext;
  bool m_sSEKMSEncryptionContextHasBeenSet = false;

  bool m_bucketKeyEnabled{false};
  bool m_bucketKeyEnabledHasBeenSet = false;

  SessionCredentials m_credentials;
  bool m_credentialsHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
