﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rekognition/RekognitionRequest.h>
#include <aws/rekognition/Rekognition_EXPORTS.h>
#include <aws/rekognition/model/NotificationChannel.h>
#include <aws/rekognition/model/StartTextDetectionFilters.h>
#include <aws/rekognition/model/Video.h>

#include <utility>

namespace Aws {
namespace Rekognition {
namespace Model {

/**
 */
class StartTextDetectionRequest : public RekognitionRequest {
 public:
  AWS_REKOGNITION_API StartTextDetectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartTextDetection"; }

  AWS_REKOGNITION_API Aws::String SerializePayload() const override;

  AWS_REKOGNITION_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{

  inline const Video& GetVideo() const { return m_video; }
  inline bool VideoHasBeenSet() const { return m_videoHasBeenSet; }
  template <typename VideoT = Video>
  void SetVideo(VideoT&& value) {
    m_videoHasBeenSet = true;
    m_video = std::forward<VideoT>(value);
  }
  template <typename VideoT = Video>
  StartTextDetectionRequest& WithVideo(VideoT&& value) {
    SetVideo(std::forward<VideoT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Idempotent token used to identify the start request. If you use the same
   * token with multiple <code>StartTextDetection</code> requests, the same
   * <code>JobId</code> is returned. Use <code>ClientRequestToken</code> to prevent
   * the same job from being accidentaly started more than once.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  StartTextDetectionRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const NotificationChannel& GetNotificationChannel() const { return m_notificationChannel; }
  inline bool NotificationChannelHasBeenSet() const { return m_notificationChannelHasBeenSet; }
  template <typename NotificationChannelT = NotificationChannel>
  void SetNotificationChannel(NotificationChannelT&& value) {
    m_notificationChannelHasBeenSet = true;
    m_notificationChannel = std::forward<NotificationChannelT>(value);
  }
  template <typename NotificationChannelT = NotificationChannel>
  StartTextDetectionRequest& WithNotificationChannel(NotificationChannelT&& value) {
    SetNotificationChannel(std::forward<NotificationChannelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An identifier returned in the completion status published by your Amazon
   * Simple Notification Service topic. For example, you can use <code>JobTag</code>
   * to group related jobs and identify them in the completion notification.</p>
   */
  inline const Aws::String& GetJobTag() const { return m_jobTag; }
  inline bool JobTagHasBeenSet() const { return m_jobTagHasBeenSet; }
  template <typename JobTagT = Aws::String>
  void SetJobTag(JobTagT&& value) {
    m_jobTagHasBeenSet = true;
    m_jobTag = std::forward<JobTagT>(value);
  }
  template <typename JobTagT = Aws::String>
  StartTextDetectionRequest& WithJobTag(JobTagT&& value) {
    SetJobTag(std::forward<JobTagT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional parameters that let you set criteria the text must meet to be
   * included in your response.</p>
   */
  inline const StartTextDetectionFilters& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = StartTextDetectionFilters>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = StartTextDetectionFilters>
  StartTextDetectionRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}
 private:
  Video m_video;

  Aws::String m_clientRequestToken;

  NotificationChannel m_notificationChannel;

  Aws::String m_jobTag;

  StartTextDetectionFilters m_filters;
  bool m_videoHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = false;
  bool m_notificationChannelHasBeenSet = false;
  bool m_jobTagHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
};

}  // namespace Model
}  // namespace Rekognition
}  // namespace Aws
