﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-messaging/ChimeSDKMessagingRequest.h>
#include <aws/chime-sdk-messaging/ChimeSDKMessaging_EXPORTS.h>
#include <aws/chime-sdk-messaging/model/ChannelMembershipType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ChimeSDKMessaging {
namespace Model {

/**
 */
class ListChannelMembershipsRequest : public ChimeSDKMessagingRequest {
 public:
  AWS_CHIMESDKMESSAGING_API ListChannelMembershipsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListChannelMemberships"; }

  AWS_CHIMESDKMESSAGING_API Aws::String SerializePayload() const override;

  AWS_CHIMESDKMESSAGING_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_CHIMESDKMESSAGING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The maximum number of channel memberships that you want returned.</p>
   */
  inline const Aws::String& GetChannelArn() const { return m_channelArn; }
  inline bool ChannelArnHasBeenSet() const { return m_channelArnHasBeenSet; }
  template <typename ChannelArnT = Aws::String>
  void SetChannelArn(ChannelArnT&& value) {
    m_channelArnHasBeenSet = true;
    m_channelArn = std::forward<ChannelArnT>(value);
  }
  template <typename ChannelArnT = Aws::String>
  ListChannelMembershipsRequest& WithChannelArn(ChannelArnT&& value) {
    SetChannelArn(std::forward<ChannelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The membership type of a user, <code>DEFAULT</code> or <code>HIDDEN</code>.
   * Default members are returned as part of <code>ListChannelMemberships</code> if
   * no type is specified. Hidden members are only returned if the type filter in
   * <code>ListChannelMemberships</code> equals <code>HIDDEN</code>.</p>
   */
  inline ChannelMembershipType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(ChannelMembershipType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline ListChannelMembershipsRequest& WithType(ChannelMembershipType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of channel memberships that you want returned.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListChannelMembershipsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token passed by previous API calls until all requested channel
   * memberships are returned.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListChannelMembershipsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the <code>AppInstanceUser</code> or <code>AppInstanceBot</code>
   * that makes the API call.</p>
   */
  inline const Aws::String& GetChimeBearer() const { return m_chimeBearer; }
  inline bool ChimeBearerHasBeenSet() const { return m_chimeBearerHasBeenSet; }
  template <typename ChimeBearerT = Aws::String>
  void SetChimeBearer(ChimeBearerT&& value) {
    m_chimeBearerHasBeenSet = true;
    m_chimeBearer = std::forward<ChimeBearerT>(value);
  }
  template <typename ChimeBearerT = Aws::String>
  ListChannelMembershipsRequest& WithChimeBearer(ChimeBearerT&& value) {
    SetChimeBearer(std::forward<ChimeBearerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the SubChannel in the request.</p>  <p>Only required when
   * listing a user's memberships in a particular sub-channel of an elastic
   * channel.</p>
   */
  inline const Aws::String& GetSubChannelId() const { return m_subChannelId; }
  inline bool SubChannelIdHasBeenSet() const { return m_subChannelIdHasBeenSet; }
  template <typename SubChannelIdT = Aws::String>
  void SetSubChannelId(SubChannelIdT&& value) {
    m_subChannelIdHasBeenSet = true;
    m_subChannelId = std::forward<SubChannelIdT>(value);
  }
  template <typename SubChannelIdT = Aws::String>
  ListChannelMembershipsRequest& WithSubChannelId(SubChannelIdT&& value) {
    SetSubChannelId(std::forward<SubChannelIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelArn;

  ChannelMembershipType m_type{ChannelMembershipType::NOT_SET};

  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::String m_chimeBearer;

  Aws::String m_subChannelId;
  bool m_channelArnHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_chimeBearerHasBeenSet = false;
  bool m_subChannelIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ChimeSDKMessaging
}  // namespace Aws
