#include "js/ForOfIterator.h"
#include "mozilla/FloatingPoint.h"
#include "mozilla/dom/BindingCallContext.h"
#include "mozilla/dom/Blob.h"
#include "mozilla/dom/CanvasGradient.h"
#include "mozilla/dom/CanvasPattern.h"
#include "mozilla/dom/Directory.h"
#include "mozilla/dom/File.h"
#include "mozilla/dom/FormData.h"
#include "mozilla/dom/HTMLCanvasElement.h"
#include "mozilla/dom/HTMLOptGroupElement.h"
#include "mozilla/dom/HTMLOptionElement.h"
#include "mozilla/dom/OffscreenCanvas.h"
#include "mozilla/dom/PrimitiveConversions.h"
#include "mozilla/dom/TrustedHTML.h"
#include "mozilla/dom/TrustedScriptURL.h"
#include "mozilla/dom/UnionTypes.h"
#include "nsGenericHTMLElement.h"
#include "nsIContent.h"

namespace mozilla::dom {
void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsBlob()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsBlob(), "mBlob", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningCanvasPatternOrCanvasGradient& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsCanvasPattern()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasPattern(), "mCanvasPattern", aFlags);
  } else if (aUnion.IsCanvasGradient()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasGradient(), "mCanvasGradient", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningCanvasPatternOrNullOrCanvasGradient& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsCanvasPattern()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasPattern(), "mCanvasPattern", aFlags);
  } else if (aUnion.IsCanvasGradient()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasGradient(), "mCanvasGradient", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningFileOrDirectory& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsFile()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsFile(), "mFile", aFlags);
  } else if (aUnion.IsDirectory()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsDirectory(), "mDirectory", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningFileOrUSVStringOrFormData& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsFile()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsFile(), "mFile", aFlags);
  } else if (aUnion.IsFormData()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsFormData(), "mFormData", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningHTMLCanvasElementOrOffscreenCanvas& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsHTMLCanvasElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLCanvasElement(), "mHTMLCanvasElement", aFlags);
  } else if (aUnion.IsOffscreenCanvas()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsOffscreenCanvas(), "mOffscreenCanvas", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningHTMLElementOrLong& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsHTMLElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLElement(), "mHTMLElement", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningHTMLOptionElementOrHTMLOptGroupElement& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsHTMLOptionElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLOptionElement(), "mHTMLOptionElement", aFlags);
  } else if (aUnion.IsHTMLOptGroupElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLOptGroupElement(), "mHTMLOptGroupElement", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningNodeOrString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsNode()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsNode(), "mNode", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedHTMLOrNullIsEmptyString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedHTML()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedHTML(), "mTrustedHTML", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedHTMLOrString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedHTML()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedHTML(), "mTrustedHTML", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedScriptURLOrString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedScriptURL()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedScriptURL(), "mTrustedScriptURL", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedScriptURLOrUSVString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedScriptURL()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedScriptURL(), "mTrustedScriptURL", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningUndefinedOrCanvasPattern& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsCanvasPattern()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasPattern(), "mCanvasPattern", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningUndefinedOrCanvasPatternOrNull& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsCanvasPattern()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasPattern(), "mCanvasPattern", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningUndefinedOrNullOrCanvasPattern& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsCanvasPattern()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsCanvasPattern(), "mCanvasPattern", aFlags);
  }
}

void
ImplCycleCollectionUnlink(OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningCanvasPatternOrCanvasGradient& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningCanvasPatternOrNullOrCanvasGradient& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningFileOrDirectory& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningFileOrUSVStringOrFormData& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningHTMLCanvasElementOrOffscreenCanvas& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningHTMLElementOrLong& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningHTMLOptionElementOrHTMLOptGroupElement& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningNodeOrString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedHTMLOrNullIsEmptyString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedHTMLOrString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedScriptURLOrString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedScriptURLOrUSVString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningUndefinedOrCanvasPattern& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningUndefinedOrCanvasPatternOrNull& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningUndefinedOrNullOrCanvasPattern& aUnion)
{
  aUnion.Uninit();
}

bool
ArrayBufferViewOrArrayBuffer::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBufferView>& memberSlot = RawSetAsArrayBufferView(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBuffer::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer");
    return false;
  }
  return true;
}

bool
ArrayBufferViewOrArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ArrayBufferViewOrArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBufferView>& memberSlot = RawSetAsArrayBufferView(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::Blob>& memberSlot = RawSetAsBlob();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Blob, mozilla::dom::Blob>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyBlob();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToBlob(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToBlob(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer, Blob");
    return false;
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eBlob: {
      if (!GetOrCreateDOMReflector(cx, mValue.mBlob.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
ArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBufferView>& memberSlot = RawSetAsArrayBufferView(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrNull::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
             (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrNull::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ArrayBufferViewOrArrayBufferOrNull::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
ByteStringOrLong::TrySetToByteString(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsByteString();
    if (!ConvertJSValueToByteString(cx, value, false, "ByteString branch of (ByteString or long)", memberSlot)) {
      return false;
    }
  }
  return true;
}

bool
ByteStringOrLong::TrySetToByteString(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToByteString(cx, value, tryNext, passedToJSImpl);
}








bool
ByteStringOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (ByteString or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
ByteStringOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToByteString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
ByteStringOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ByteStringOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eByteString: {
      if (!NonVoidByteStringToJsval(cx, mValue.mByteString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
CanvasPatternOrCanvasGradient::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
CanvasPatternOrCanvasGradient::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}







bool
CanvasPatternOrCanvasGradient::TrySetToCanvasGradient(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasGradient>& memberSlot = RawSetAsCanvasGradient();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasGradient, mozilla::dom::CanvasGradient>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasGradient();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
CanvasPatternOrCanvasGradient::TrySetToCanvasGradient(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl);
}







bool
CanvasPatternOrCanvasGradient::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern, CanvasGradient");
    return false;
  }
  return true;
}

bool
CanvasPatternOrCanvasGradient::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
CanvasPatternOrCanvasGradient::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eCanvasGradient: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasGradient.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
CanvasPatternOrNullOrCanvasGradient::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
CanvasPatternOrNullOrCanvasGradient::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}







bool
CanvasPatternOrNullOrCanvasGradient::TrySetToCanvasGradient(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasGradient>& memberSlot = RawSetAsCanvasGradient();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasGradient, mozilla::dom::CanvasGradient>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasGradient();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
CanvasPatternOrNullOrCanvasGradient::TrySetToCanvasGradient(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl);
}







bool
CanvasPatternOrNullOrCanvasGradient::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
             (failed = !TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern, CanvasGradient");
      return false;
    }
  }
  return true;
}

bool
CanvasPatternOrNullOrCanvasGradient::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
CanvasPatternOrNullOrCanvasGradient::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eCanvasGradient: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasGradient.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
DoubleOrByteString::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or ByteString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or ByteString)");
      return false;
    }
  }
  return true;
}

bool
DoubleOrByteString::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}







bool
DoubleOrByteString::TrySetToByteString(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsByteString();
    if (!ConvertJSValueToByteString(cx, value, false, "ByteString branch of (double or ByteString)", memberSlot)) {
      return false;
    }
  }
  return true;
}

bool
DoubleOrByteString::TrySetToByteString(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToByteString(cx, value, tryNext, passedToJSImpl);
}








bool
DoubleOrByteString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToByteString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
DoubleOrByteString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
DoubleOrByteString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eByteString: {
      if (!NonVoidByteStringToJsval(cx, mValue.mByteString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
DoubleOrString::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or DOMString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or DOMString)");
      return false;
    }
  }
  return true;
}

bool
DoubleOrString::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}







bool
DoubleOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
DoubleOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
DoubleOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
DoubleOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
DoubleOrSupportedType::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or SupportedType)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or SupportedType)");
      return false;
    }
  }
  return true;
}

bool
DoubleOrSupportedType::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}







bool
DoubleOrSupportedType::TrySetToSupportedType(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    SupportedType& memberSlot = RawSetAsSupportedType();
    {
      int index;
      if (!binding_detail::FindEnumStringIndex<true>(cx, value,
                                                                         binding_detail::EnumStrings<SupportedType>::Values,
                                                                         "SupportedType", "SupportedType branch of (double or SupportedType)",
                                                                         &index)) {
        return false;
      }
      MOZ_ASSERT(index >= 0);
      memberSlot = static_cast<SupportedType>(index);
    }
  }
  return true;
}

bool
DoubleOrSupportedType::TrySetToSupportedType(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToSupportedType(cx, value, tryNext, passedToJSImpl);
}







bool
DoubleOrSupportedType::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToSupportedType(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
DoubleOrSupportedType::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
DoubleOrSupportedType::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eSupportedType: {
      if (!ToJSValue(cx, mValue.mSupportedType.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
DoubleOrUSVString::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or USVString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or USVString)");
      return false;
    }
  }
  return true;
}

bool
DoubleOrUSVString::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}







bool
DoubleOrUSVString::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}








bool
DoubleOrUSVString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
DoubleOrUSVString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
DoubleOrUSVString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
DoubleOrUTF8String::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or USVString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or USVString)");
      return false;
    }
  }
  return true;
}

bool
DoubleOrUTF8String::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}







bool
DoubleOrUTF8String::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
DoubleOrUTF8String::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
DoubleOrUTF8String::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
DoubleOrUTF8String::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
FileOrDirectory::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrDirectory::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrDirectory::TrySetToDirectory(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::Directory>& memberSlot = RawSetAsDirectory();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Directory, mozilla::dom::Directory>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyDirectory();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrDirectory::TrySetToDirectory(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDirectory(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrDirectory::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToDirectory(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, Directory");
    return false;
  }
  return true;
}

bool
FileOrDirectory::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
FileOrDirectory::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eDirectory: {
      if (!GetOrCreateDOMReflector(cx, mValue.mDirectory.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
FileOrUSVStringOrFormData::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrUSVStringOrFormData::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrUSVStringOrFormData::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}








bool
FileOrUSVStringOrFormData::TrySetToFormData(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::FormData>& memberSlot = RawSetAsFormData();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::FormData, mozilla::dom::FormData>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFormData();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrUSVStringOrFormData::TrySetToFormData(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFormData(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrUSVStringOrFormData::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToFormData(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, FormData");
    return false;
  }
  return true;
}

bool
FileOrUSVStringOrFormData::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
FileOrUSVStringOrFormData::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eFormData: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFormData.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
FloatOrString::TrySetToFloat(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    float& memberSlot = RawSetAsFloat();
    if (!ValueToPrimitive<float, eDefault>(cx, value, "Float branch of (float or DOMString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Float branch of (float or DOMString)");
      return false;
    }
  }
  return true;
}

bool
FloatOrString::TrySetToFloat(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFloat(cx, value, tryNext, passedToJSImpl);
}







bool
FloatOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
FloatOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToFloat(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
FloatOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
FloatOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFloat: {
      rval.set(JS_NumberValue(double(mValue.mFloat.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::HTMLCanvasElement>& memberSlot = RawSetAsHTMLCanvasElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLCanvasElement, mozilla::dom::HTMLCanvasElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLCanvasElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::OffscreenCanvas>& memberSlot = RawSetAsOffscreenCanvas();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::OffscreenCanvas, mozilla::dom::OffscreenCanvas>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyOffscreenCanvas();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLCanvasElementOrOffscreenCanvas::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLCanvasElement, OffscreenCanvas");
    return false;
  }
  return true;
}

bool
HTMLCanvasElementOrOffscreenCanvas::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
HTMLCanvasElementOrOffscreenCanvas::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLCanvasElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLCanvasElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eOffscreenCanvas: {
      if (!GetOrCreateDOMReflector(cx, mValue.mOffscreenCanvas.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
HTMLElementOrLong::TrySetToHTMLElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<nsGenericHTMLElement>& memberSlot = RawSetAsHTMLElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLElement, nsGenericHTMLElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLElementOrLong::TrySetToHTMLElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLElementOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (HTMLElement or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
HTMLElementOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLElement");
    return false;
  }
  return true;
}

bool
HTMLElementOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
HTMLElementOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::HTMLOptionElement>& memberSlot = RawSetAsHTMLOptionElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptionElement, mozilla::dom::HTMLOptionElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptionElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::HTMLOptGroupElement>& memberSlot = RawSetAsHTMLOptGroupElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptGroupElement, mozilla::dom::HTMLOptGroupElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptGroupElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLOptionElementOrHTMLOptGroupElement::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLOptionElement, HTMLOptGroupElement");
    return false;
  }
  return true;
}

bool
HTMLOptionElementOrHTMLOptGroupElement::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
HTMLOptionElementOrHTMLOptGroupElement::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLOptionElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptionElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eHTMLOptGroupElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptGroupElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
LongOrStringAnyRecord::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (long or record<DOMString, any>)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
LongOrStringAnyRecord::TrySetToStringAnyRecord(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedRecord<nsString, JS::Value>& memberSlot = RawSetAsStringAnyRecord(cx);
    auto& recordEntries = memberSlot.Entries();

    JS::Rooted<JSObject*> recordObj(cx, &value.toObject());
    JS::RootedVector<jsid> ids(cx);
    if (!js::GetPropertyKeys(cx, recordObj,
                             JSITER_OWNONLY | JSITER_HIDDEN | JSITER_SYMBOLS, &ids)) {
      return false;
    }
    if (!recordEntries.SetCapacity(ids.length(), mozilla::fallible)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
    JS::Rooted<JS::Value> propNameValue(cx);
    JS::Rooted<JS::Value> temp(cx);
    JS::Rooted<jsid> curId(cx);
    JS::Rooted<JS::Value> idVal(cx);
    // Use a hashset to keep track of ids seen, to avoid
    // introducing nasty O(N^2) behavior scanning for them all the
    // time.  Ideally we'd use a data structure with O(1) lookup
    // _and_ ordering for the MozMap, but we don't have one lying
    // around.
    nsTHashtable<nsStringHashKey> idsSeen;
    for (size_t i = 0; i < ids.length(); ++i) {
      curId = ids[i];

      JS::Rooted<mozilla::Maybe<JS::PropertyDescriptor>> desc(cx);
      if (!JS_GetOwnPropertyDescriptorById(cx, recordObj, curId,
                                           &desc)) {
        return false;
      }

      if (desc.isNothing() || !desc->enumerable()) {
        continue;
      }

      idVal = js::IdToValue(curId);
      nsString propName;
      // This will just throw if idVal is a Symbol, like the spec says
      // to do.
      if (!ConvertJSValueToString(cx, idVal, "key of record<DOMString, any> branch of (long or record<DOMString, any>)", propName)) {
        return false;
      }

      if (!JS_GetPropertyById(cx, recordObj, curId, &temp)) {
        return false;
      }

      Record<nsString, JS::Value>::EntryType* entry;
      if (!idsSeen.EnsureInserted(propName)) {
        // Find the existing entry.
        auto idx = recordEntries.IndexOf(propName);
        MOZ_ASSERT(idx != recordEntries.NoIndex,
                   "Why is it not found?");
        // Now blow it away to make it look like it was just added
        // to the array, because it's not obvious that it's
        // safe to write to its already-initialized mValue via our
        // normal codegen conversions.  For example, the value
        // could be a union and this would change its type, but
        // codegen assumes we won't do that.
        entry = recordEntries.ReconstructElementAt(idx);
      } else {
        // Safe to do an infallible append here, because we did a
        // SetCapacity above to the right capacity.
        entry = recordEntries.AppendElement();
      }
      entry->mKey = propName;
      JS::Value& slot = entry->mValue;
#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunreachable-code"
#pragma clang diagnostic ignored "-Wunreachable-code-return"
#endif // __clang__
      if ((passedToJSImpl) && !CallerSubsumes(temp)) {
        cx.ThrowErrorMessage<MSG_PERMISSION_DENIED_TO_PASS_ARG>("value in record<DOMString, any> branch of (long or record<DOMString, any>)");
        return false;
      }
#ifdef __clang__
#pragma clang diagnostic pop
#endif // __clang__
      slot = temp;
    }
  }
  return true;
}

bool
LongOrStringAnyRecord::TrySetToStringAnyRecord(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToStringAnyRecord(cx, value, tryNext, passedToJSImpl);
}







bool
LongOrStringAnyRecord::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!done) {
      done = (failed = !TrySetToStringAnyRecord(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
  }
  if (!done) {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "record<DOMString, any>");
    return false;
  }
  return true;
}

bool
LongOrStringAnyRecord::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
LongOrStringAnyRecord::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    case eStringAnyRecord: {

      JS::Rooted<JSObject*> returnObj(cx, JS_NewPlainObject(cx));
      if (!returnObj) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (auto& entry : mValue.mStringAnyRecord.Value().Entries()) {
          auto& recordValue0 = entry.mValue;
          // Control block to let us common up the JS_DefineUCProperty calls when there
          // are different ways to succeed at wrapping the value.
          do {
            JS::ExposeValueToActiveJS(recordValue0);
            tmp.set(recordValue0);
            if (!MaybeWrapValue(cx, &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineUCProperty(cx, returnObj,
                                   entry.mKey.BeginReading(),
                                   entry.mKey.Length(), tmp,
                                   JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnObj);
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt8Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<Int8Array>& memberSlot = RawSetAsInt8Array(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyInt8Array();
      tryNext = true;
      return true;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Int8Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Int8Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Int8Array branch of (Int8Array or Int16Array)");
      return false;
    }
  }
  return true;
}

bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt8Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToInt8Array(cx, value, tryNext, passedToJSImpl);
}







bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt16Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<Int16Array>& memberSlot = RawSetAsInt16Array(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyInt16Array();
      tryNext = true;
      return true;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Int16Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Int16Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Int16Array branch of (Int8Array or Int16Array)");
      return false;
    }
  }
  return true;
}

bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt16Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToInt16Array(cx, value, tryNext, passedToJSImpl);
}







bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToInt8Array(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToInt16Array(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Int8Array, Int16Array");
    return false;
  }
  return true;
}

bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
MaybeSharedInt8ArrayOrMaybeSharedInt16Array::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eInt8Array: {
      rval.setObject(*mValue.mInt8Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eInt16Array: {
      rval.setObject(*mValue.mInt16Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
NodeOrString::TrySetToNode(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<nsINode>& memberSlot = RawSetAsNode();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Node, nsINode>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyNode();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
NodeOrString::TrySetToNode(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToNode(cx, value, tryNext, passedToJSImpl);
}







bool
NodeOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
NodeOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToNode(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Node");
    return false;
  }
  return true;
}

bool
NodeOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
NodeOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNode: {
      if (!GetOrCreateDOMReflector(cx, mValue.mNode.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
ObjectOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (object or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
ObjectOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
ObjectOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ObjectOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
ObjectOrNullOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (object? or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
ObjectOrNullOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
        return false;
      }
      done = true;
    } else {
      do {
        done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
        break;
      } while (false);
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
      return false;
    }
  }
  return true;
}

bool
ObjectOrNullOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ObjectOrNullOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrArrayBuffer::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
StringOrArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
StringOrArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
StringOrArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
    return false;
  }
  return true;
}

bool
StringOrArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrBooleanOrObject::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
StringOrBooleanOrObject::TrySetToBoolean(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    bool& memberSlot = RawSetAsBoolean();
    if (!ValueToPrimitive<bool, eDefault>(cx, value, "Boolean branch of (DOMString or boolean or object)", &memberSlot)) {
      return false;
    }
  }
  return true;
}












bool
StringOrBooleanOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      if (value.isBoolean()) {
        done = (failed = !TrySetToBoolean(cx, value, tryNext)) || !tryNext;
        break;
      }
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
StringOrBooleanOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrBooleanOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eBoolean: {
      rval.setBoolean(mValue.mBoolean.Value());
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrMaybeSharedArrayBuffer::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
StringOrMaybeSharedArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
StringOrMaybeSharedArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
StringOrMaybeSharedArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
    return false;
  }
  return true;
}

bool
StringOrMaybeSharedArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrMaybeSharedArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrObject::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}













bool
StringOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
StringOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrStringSequence::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
StringOrStringSequence::TrySetToStringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::AutoSequence<nsString>& memberSlot = RawSetAsStringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyStringSequence();
      tryNext = true;
      return true;
    }
    binding_detail::AutoSequence<nsString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
StringOrStringSequence::TrySetToStringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToStringSequence(cx, value, tryNext, passedToJSImpl);
}







bool
StringOrStringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToStringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<DOMString>");
    return false;
  }
  return true;
}

bool
StringOrStringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrStringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eStringSequence: {

      uint32_t length = mValue.mStringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!xpc::NonVoidStringToJsval(cx, mValue.mStringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
SupportedTypeOrObject::TrySetToSupportedType(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    SupportedType& memberSlot = RawSetAsSupportedType();
    {
      int index;
      if (!binding_detail::FindEnumStringIndex<true>(cx, value,
                                                                         binding_detail::EnumStrings<SupportedType>::Values,
                                                                         "SupportedType", "SupportedType branch of (SupportedType or object)",
                                                                         &index)) {
        return false;
      }
      MOZ_ASSERT(index >= 0);
      memberSlot = static_cast<SupportedType>(index);
    }
  }
  return true;
}

bool
SupportedTypeOrObject::TrySetToSupportedType(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToSupportedType(cx, value, tryNext, passedToJSImpl);
}












bool
SupportedTypeOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      done = (failed = !TrySetToSupportedType(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
SupportedTypeOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
SupportedTypeOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eSupportedType: {
      if (!ToJSValue(cx, mValue.mSupportedType.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedHTMLOrNullIsEmptyString::TrySetToNullIsEmptyString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsNullIsEmptyString();
    if (!ConvertJSValueToString(cx, value, eEmpty, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
TrustedHTMLOrNullIsEmptyString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToNullIsEmptyString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
TrustedHTMLOrNullIsEmptyString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedHTMLOrNullIsEmptyString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eNullIsEmptyString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mNullIsEmptyString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedHTMLOrString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedHTMLOrString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedHTMLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
TrustedHTMLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
TrustedHTMLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedHTMLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedScriptURLOrString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedScriptURLOrString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedScriptURLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
TrustedScriptURLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
TrustedScriptURLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedScriptURLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedScriptURLOrUSVString::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}








bool
TrustedScriptURLOrUSVString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
TrustedScriptURLOrUSVString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedScriptURLOrUSVString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrArrayBuffer::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
UTF8StringOrArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
UTF8StringOrArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
    return false;
  }
  return true;
}

bool
UTF8StringOrArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrArrayBufferOrNull::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrArrayBufferOrNull::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
  }
  return true;
}

bool
UTF8StringOrArrayBufferOrNull::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
UTF8StringOrArrayBufferOrNull::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (!done) {
      do {
        done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
        break;
      } while (false);
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
      return false;
    }
  }
  return true;
}

bool
UTF8StringOrArrayBufferOrNull::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrArrayBufferOrNull::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrLong::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (USVString or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
UTF8StringOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
UTF8StringOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrUTF8StringSequence::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::AutoSequence<nsCString>& memberSlot = RawSetAsUTF8StringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyUTF8StringSequence();
      tryNext = true;
      return true;
    }
    binding_detail::AutoSequence<nsCString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsCString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsCString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
UTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl);
}







bool
UTF8StringOrUTF8StringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<USVString>");
    return false;
  }
  return true;
}

bool
UTF8StringOrUTF8StringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrUTF8StringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUTF8StringSequence: {

      uint32_t length = mValue.mUTF8StringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8StringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrUint8Array::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrUint8Array::TrySetToUint8Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<Uint8Array>& memberSlot = RawSetAsUint8Array(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyUint8Array();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
  }
  return true;
}

bool
UTF8StringOrUint8Array::TrySetToUint8Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUint8Array(cx, value, tryNext, passedToJSImpl);
}







bool
UTF8StringOrUint8Array::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUint8Array(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Uint8Array");
    return false;
  }
  return true;
}

bool
UTF8StringOrUint8Array::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrUint8Array::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUint8Array: {
      rval.setObject(*mValue.mUint8Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UndefinedOrCanvasPattern::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
UndefinedOrCanvasPattern::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}







bool
UndefinedOrCanvasPattern::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isUndefined()) {
    SetUndefined();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern");
      return false;
    }
  }
  return true;
}

bool
UndefinedOrCanvasPattern::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UndefinedOrCanvasPattern::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUndefined: {
      rval.setUndefined();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UndefinedOrCanvasPatternOrNull::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
UndefinedOrCanvasPatternOrNull::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}







bool
UndefinedOrCanvasPatternOrNull::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isUndefined()) {
    SetUndefined();
  } else if (value.isNull()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern");
      return false;
    }
  }
  return true;
}

bool
UndefinedOrCanvasPatternOrNull::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UndefinedOrCanvasPatternOrNull::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eUndefined: {
      rval.setUndefined();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UndefinedOrNullOrCanvasPattern::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
UndefinedOrNullOrCanvasPattern::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}







bool
UndefinedOrNullOrCanvasPattern::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isUndefined()) {
    SetUndefined();
  } else if (value.isNull()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern");
      return false;
    }
  }
  return true;
}

bool
UndefinedOrNullOrCanvasPattern::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UndefinedOrNullOrCanvasPattern::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eUndefined: {
      rval.setUndefined();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UnrestrictedDoubleOrString::TrySetToUnrestrictedDouble(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsUnrestrictedDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Unrestricted double branch of (unrestricted double or DOMString)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
UnrestrictedDoubleOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UnrestrictedDoubleOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToUnrestrictedDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
UnrestrictedDoubleOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UnrestrictedDoubleOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUnrestrictedDouble: {
      rval.set(JS_NumberValue(double(mValue.mUnrestrictedDouble.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UnrestrictedFloatOrString::TrySetToUnrestrictedFloat(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    float& memberSlot = RawSetAsUnrestrictedFloat();
    if (!ValueToPrimitive<float, eDefault>(cx, value, "Unrestricted float branch of (unrestricted float or DOMString)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
UnrestrictedFloatOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UnrestrictedFloatOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToUnrestrictedFloat(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
UnrestrictedFloatOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UnrestrictedFloatOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUnrestrictedFloat: {
      rval.set(JS_NumberValue(double(mValue.mUnrestrictedFloat.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}


OwningArrayBufferViewOrArrayBuffer::OwningArrayBufferViewOrArrayBuffer(OwningArrayBufferViewOrArrayBuffer&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eArrayBufferView: {
      mType = eArrayBufferView;
      mValue.mArrayBufferView.SetValue(std::move(aOther.mValue.mArrayBufferView.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
  }
}


bool
OwningArrayBufferViewOrArrayBuffer::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBufferView& memberSlot = RawSetAsArrayBufferView();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBuffer::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBuffer::RawSetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBuffer::SetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  Uninit();
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}


void
OwningArrayBufferViewOrArrayBuffer::DestroyArrayBufferView()
{
  MOZ_RELEASE_ASSERT(IsArrayBufferView(), "Wrong type!");
  mValue.mArrayBufferView.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBuffer::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBuffer::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningArrayBufferViewOrArrayBuffer::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer");
    return false;
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningArrayBufferViewOrArrayBuffer::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eArrayBufferView: {
      DestroyArrayBufferView();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
  }
}

bool
OwningArrayBufferViewOrArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningArrayBufferViewOrArrayBuffer::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBufferView: {
      mValue.mArrayBufferView.Value().TraceSelf(trc);
      break;
    }
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningArrayBufferViewOrArrayBuffer&
OwningArrayBufferViewOrArrayBuffer::operator=(OwningArrayBufferViewOrArrayBuffer&& aOther)
{
  this->~OwningArrayBufferViewOrArrayBuffer();
  new (this) OwningArrayBufferViewOrArrayBuffer (std::move(aOther));
  return *this;
}



OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String(OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eArrayBufferView: {
      mType = eArrayBufferView;
      mValue.mArrayBufferView.SetValue(std::move(aOther.mValue.mArrayBufferView.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
    case eBlob: {
      mType = eBlob;
      mValue.mBlob.SetValue(std::move(aOther.mValue.mBlob.Value()));
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
  }
}


bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBufferView& memberSlot = RawSetAsArrayBufferView();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  Uninit();
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyArrayBufferView()
{
  MOZ_RELEASE_ASSERT(IsArrayBufferView(), "Wrong type!");
  mValue.mArrayBufferView.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::Blob>& memberSlot = RawSetAsBlob();
    static_assert(IsRefcounted<mozilla::dom::Blob>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Blob, mozilla::dom::Blob>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyBlob();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToBlob(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::Blob>&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsBlob()
{
  if (mType == eBlob) {
    return mValue.mBlob.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eBlob;
  return mValue.mBlob.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::Blob>&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsBlob()
{
  if (mType == eBlob) {
    return mValue.mBlob.Value();
  }
  Uninit();
  mType = eBlob;
  return mValue.mBlob.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyBlob()
{
  MOZ_RELEASE_ASSERT(IsBlob(), "Wrong type!");
  mValue.mBlob.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToBlob(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer, Blob");
    return false;
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eArrayBufferView: {
      DestroyArrayBufferView();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
    case eBlob: {
      DestroyBlob();
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
  }
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eBlob: {
      if (!GetOrCreateDOMReflector(cx, mValue.mBlob.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBufferView: {
      mValue.mArrayBufferView.Value().TraceSelf(trc);
      break;
    }
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::operator=(OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String&& aOther)
{
  this->~OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String();
  new (this) OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String (std::move(aOther));
  return *this;
}



OwningArrayBufferViewOrArrayBufferOrNull::OwningArrayBufferViewOrArrayBufferOrNull(OwningArrayBufferViewOrArrayBufferOrNull&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eArrayBufferView: {
      mType = eArrayBufferView;
      mValue.mArrayBufferView.SetValue(std::move(aOther.mValue.mArrayBufferView.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
  }
}




bool
OwningArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBufferView& memberSlot = RawSetAsArrayBufferView();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBufferOrNull::RawSetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBufferOrNull::SetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  Uninit();
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrNull::DestroyArrayBufferView()
{
  MOZ_RELEASE_ASSERT(IsArrayBufferView(), "Wrong type!");
  mValue.mArrayBufferView.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (ArrayBufferView or ArrayBuffer?)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrNull::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBufferOrNull::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBufferOrNull::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrNull::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrNull::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
             (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrNull::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningArrayBufferViewOrArrayBufferOrNull::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNull: {
      break;
    }
    case eArrayBufferView: {
      DestroyArrayBufferView();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
  }
}

bool
OwningArrayBufferViewOrArrayBufferOrNull::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningArrayBufferViewOrArrayBufferOrNull::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBufferView: {
      mValue.mArrayBufferView.Value().TraceSelf(trc);
      break;
    }
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningArrayBufferViewOrArrayBufferOrNull&
OwningArrayBufferViewOrArrayBufferOrNull::operator=(OwningArrayBufferViewOrArrayBufferOrNull&& aOther)
{
  this->~OwningArrayBufferViewOrArrayBufferOrNull();
  new (this) OwningArrayBufferViewOrArrayBufferOrNull (std::move(aOther));
  return *this;
}



OwningByteStringOrLong::OwningByteStringOrLong(OwningByteStringOrLong&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eByteString: {
      mType = eByteString;
      mValue.mByteString.SetValue(std::move(aOther.mValue.mByteString.Value()));
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
  }
}



bool
OwningByteStringOrLong::TrySetToByteString(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsByteString();
    if (!ConvertJSValueToByteString(cx, value, false, "ByteString branch of (ByteString or long)", memberSlot)) {
      return false;
    }
  }
  return true;
}

bool
OwningByteStringOrLong::TrySetToByteString(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToByteString(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] nsCString&
OwningByteStringOrLong::RawSetAsByteString()
{
  if (mType == eByteString) {
    return mValue.mByteString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eByteString;
  return mValue.mByteString.SetValue();
}

[[nodiscard]] nsCString&
OwningByteStringOrLong::SetAsByteString()
{
  if (mType == eByteString) {
    return mValue.mByteString.Value();
  }
  Uninit();
  mType = eByteString;
  return mValue.mByteString.SetValue();
}



void
OwningByteStringOrLong::DestroyByteString()
{
  MOZ_RELEASE_ASSERT(IsByteString(), "Wrong type!");
  mValue.mByteString.Destroy();
  mType = eUninitialized;
}



bool
OwningByteStringOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (ByteString or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningByteStringOrLong::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningByteStringOrLong::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningByteStringOrLong::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningByteStringOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToByteString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningByteStringOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningByteStringOrLong::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eByteString: {
      DestroyByteString();
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
  }
}

bool
OwningByteStringOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eByteString: {
      if (!NonVoidByteStringToJsval(cx, mValue.mByteString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningByteStringOrLong&
OwningByteStringOrLong::operator=(OwningByteStringOrLong&& aOther)
{
  this->~OwningByteStringOrLong();
  new (this) OwningByteStringOrLong (std::move(aOther));
  return *this;
}


OwningByteStringOrLong&
OwningByteStringOrLong::operator=(const OwningByteStringOrLong& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eByteString: {
      SetAsByteString() = aOther.GetAsByteString();
      break;
    }
    case eLong: {
      SetAsLong() = aOther.GetAsLong();
      break;
    }
  }
  return *this;
}


OwningCanvasPatternOrCanvasGradient::OwningCanvasPatternOrCanvasGradient(OwningCanvasPatternOrCanvasGradient&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eCanvasPattern: {
      mType = eCanvasPattern;
      mValue.mCanvasPattern.SetValue(std::move(aOther.mValue.mCanvasPattern.Value()));
      break;
    }
    case eCanvasGradient: {
      mType = eCanvasGradient;
      mValue.mCanvasGradient.SetValue(std::move(aOther.mValue.mCanvasGradient.Value()));
      break;
    }
  }
}



bool
OwningCanvasPatternOrCanvasGradient::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    static_assert(IsRefcounted<mozilla::dom::CanvasPattern>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningCanvasPatternOrCanvasGradient::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningCanvasPatternOrCanvasGradient::RawSetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningCanvasPatternOrCanvasGradient::SetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  Uninit();
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}


void
OwningCanvasPatternOrCanvasGradient::DestroyCanvasPattern()
{
  MOZ_RELEASE_ASSERT(IsCanvasPattern(), "Wrong type!");
  mValue.mCanvasPattern.Destroy();
  mType = eUninitialized;
}



bool
OwningCanvasPatternOrCanvasGradient::TrySetToCanvasGradient(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasGradient>& memberSlot = RawSetAsCanvasGradient();
    static_assert(IsRefcounted<mozilla::dom::CanvasGradient>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasGradient, mozilla::dom::CanvasGradient>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasGradient();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningCanvasPatternOrCanvasGradient::TrySetToCanvasGradient(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasGradient>&
OwningCanvasPatternOrCanvasGradient::RawSetAsCanvasGradient()
{
  if (mType == eCanvasGradient) {
    return mValue.mCanvasGradient.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasGradient;
  return mValue.mCanvasGradient.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasGradient>&
OwningCanvasPatternOrCanvasGradient::SetAsCanvasGradient()
{
  if (mType == eCanvasGradient) {
    return mValue.mCanvasGradient.Value();
  }
  Uninit();
  mType = eCanvasGradient;
  return mValue.mCanvasGradient.SetValue();
}


void
OwningCanvasPatternOrCanvasGradient::DestroyCanvasGradient()
{
  MOZ_RELEASE_ASSERT(IsCanvasGradient(), "Wrong type!");
  mValue.mCanvasGradient.Destroy();
  mType = eUninitialized;
}



bool
OwningCanvasPatternOrCanvasGradient::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern, CanvasGradient");
    return false;
  }
  return true;
}

bool
OwningCanvasPatternOrCanvasGradient::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningCanvasPatternOrCanvasGradient::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eCanvasPattern: {
      DestroyCanvasPattern();
      break;
    }
    case eCanvasGradient: {
      DestroyCanvasGradient();
      break;
    }
  }
}

bool
OwningCanvasPatternOrCanvasGradient::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eCanvasGradient: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasGradient.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningCanvasPatternOrCanvasGradient&
OwningCanvasPatternOrCanvasGradient::operator=(OwningCanvasPatternOrCanvasGradient&& aOther)
{
  this->~OwningCanvasPatternOrCanvasGradient();
  new (this) OwningCanvasPatternOrCanvasGradient (std::move(aOther));
  return *this;
}


OwningCanvasPatternOrCanvasGradient&
OwningCanvasPatternOrCanvasGradient::operator=(const OwningCanvasPatternOrCanvasGradient& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eCanvasPattern: {
      SetAsCanvasPattern() = aOther.GetAsCanvasPattern();
      break;
    }
    case eCanvasGradient: {
      SetAsCanvasGradient() = aOther.GetAsCanvasGradient();
      break;
    }
  }
  return *this;
}


OwningCanvasPatternOrNullOrCanvasGradient::OwningCanvasPatternOrNullOrCanvasGradient(OwningCanvasPatternOrNullOrCanvasGradient&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eCanvasPattern: {
      mType = eCanvasPattern;
      mValue.mCanvasPattern.SetValue(std::move(aOther.mValue.mCanvasPattern.Value()));
      break;
    }
    case eCanvasGradient: {
      mType = eCanvasGradient;
      mValue.mCanvasGradient.SetValue(std::move(aOther.mValue.mCanvasGradient.Value()));
      break;
    }
  }
}





bool
OwningCanvasPatternOrNullOrCanvasGradient::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    static_assert(IsRefcounted<mozilla::dom::CanvasPattern>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningCanvasPatternOrNullOrCanvasGradient::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningCanvasPatternOrNullOrCanvasGradient::RawSetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningCanvasPatternOrNullOrCanvasGradient::SetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  Uninit();
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}


void
OwningCanvasPatternOrNullOrCanvasGradient::DestroyCanvasPattern()
{
  MOZ_RELEASE_ASSERT(IsCanvasPattern(), "Wrong type!");
  mValue.mCanvasPattern.Destroy();
  mType = eUninitialized;
}



bool
OwningCanvasPatternOrNullOrCanvasGradient::TrySetToCanvasGradient(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasGradient>& memberSlot = RawSetAsCanvasGradient();
    static_assert(IsRefcounted<mozilla::dom::CanvasGradient>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasGradient, mozilla::dom::CanvasGradient>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasGradient();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningCanvasPatternOrNullOrCanvasGradient::TrySetToCanvasGradient(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasGradient>&
OwningCanvasPatternOrNullOrCanvasGradient::RawSetAsCanvasGradient()
{
  if (mType == eCanvasGradient) {
    return mValue.mCanvasGradient.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasGradient;
  return mValue.mCanvasGradient.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasGradient>&
OwningCanvasPatternOrNullOrCanvasGradient::SetAsCanvasGradient()
{
  if (mType == eCanvasGradient) {
    return mValue.mCanvasGradient.Value();
  }
  Uninit();
  mType = eCanvasGradient;
  return mValue.mCanvasGradient.SetValue();
}


void
OwningCanvasPatternOrNullOrCanvasGradient::DestroyCanvasGradient()
{
  MOZ_RELEASE_ASSERT(IsCanvasGradient(), "Wrong type!");
  mValue.mCanvasGradient.Destroy();
  mType = eUninitialized;
}



bool
OwningCanvasPatternOrNullOrCanvasGradient::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
             (failed = !TrySetToCanvasGradient(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern, CanvasGradient");
      return false;
    }
  }
  return true;
}

bool
OwningCanvasPatternOrNullOrCanvasGradient::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningCanvasPatternOrNullOrCanvasGradient::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNull: {
      break;
    }
    case eCanvasPattern: {
      DestroyCanvasPattern();
      break;
    }
    case eCanvasGradient: {
      DestroyCanvasGradient();
      break;
    }
  }
}

bool
OwningCanvasPatternOrNullOrCanvasGradient::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eCanvasGradient: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasGradient.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningCanvasPatternOrNullOrCanvasGradient&
OwningCanvasPatternOrNullOrCanvasGradient::operator=(OwningCanvasPatternOrNullOrCanvasGradient&& aOther)
{
  this->~OwningCanvasPatternOrNullOrCanvasGradient();
  new (this) OwningCanvasPatternOrNullOrCanvasGradient (std::move(aOther));
  return *this;
}


OwningCanvasPatternOrNullOrCanvasGradient&
OwningCanvasPatternOrNullOrCanvasGradient::operator=(const OwningCanvasPatternOrNullOrCanvasGradient& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eCanvasPattern: {
      SetAsCanvasPattern() = aOther.GetAsCanvasPattern();
      break;
    }
    case eCanvasGradient: {
      SetAsCanvasGradient() = aOther.GetAsCanvasGradient();
      break;
    }
  }
  return *this;
}


OwningDoubleOrByteString::OwningDoubleOrByteString(OwningDoubleOrByteString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      mType = eDouble;
      mValue.mDouble.SetValue(std::move(aOther.mValue.mDouble.Value()));
      break;
    }
    case eByteString: {
      mType = eByteString;
      mValue.mByteString.SetValue(std::move(aOther.mValue.mByteString.Value()));
      break;
    }
  }
}



bool
OwningDoubleOrByteString::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or ByteString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or ByteString)");
      return false;
    }
  }
  return true;
}

bool
OwningDoubleOrByteString::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] double&
OwningDoubleOrByteString::RawSetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDouble;
  return mValue.mDouble.SetValue();
}

[[nodiscard]] double&
OwningDoubleOrByteString::SetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  Uninit();
  mType = eDouble;
  return mValue.mDouble.SetValue();
}


void
OwningDoubleOrByteString::DestroyDouble()
{
  MOZ_RELEASE_ASSERT(IsDouble(), "Wrong type!");
  mValue.mDouble.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrByteString::TrySetToByteString(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsByteString();
    if (!ConvertJSValueToByteString(cx, value, false, "ByteString branch of (double or ByteString)", memberSlot)) {
      return false;
    }
  }
  return true;
}

bool
OwningDoubleOrByteString::TrySetToByteString(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToByteString(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] nsCString&
OwningDoubleOrByteString::RawSetAsByteString()
{
  if (mType == eByteString) {
    return mValue.mByteString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eByteString;
  return mValue.mByteString.SetValue();
}

[[nodiscard]] nsCString&
OwningDoubleOrByteString::SetAsByteString()
{
  if (mType == eByteString) {
    return mValue.mByteString.Value();
  }
  Uninit();
  mType = eByteString;
  return mValue.mByteString.SetValue();
}



void
OwningDoubleOrByteString::DestroyByteString()
{
  MOZ_RELEASE_ASSERT(IsByteString(), "Wrong type!");
  mValue.mByteString.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrByteString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToByteString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningDoubleOrByteString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningDoubleOrByteString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eDouble: {
      DestroyDouble();
      break;
    }
    case eByteString: {
      DestroyByteString();
      break;
    }
  }
}

bool
OwningDoubleOrByteString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eByteString: {
      if (!NonVoidByteStringToJsval(cx, mValue.mByteString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningDoubleOrByteString&
OwningDoubleOrByteString::operator=(OwningDoubleOrByteString&& aOther)
{
  this->~OwningDoubleOrByteString();
  new (this) OwningDoubleOrByteString (std::move(aOther));
  return *this;
}


OwningDoubleOrByteString&
OwningDoubleOrByteString::operator=(const OwningDoubleOrByteString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      SetAsDouble() = aOther.GetAsDouble();
      break;
    }
    case eByteString: {
      SetAsByteString() = aOther.GetAsByteString();
      break;
    }
  }
  return *this;
}


OwningDoubleOrString::OwningDoubleOrString(OwningDoubleOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      mType = eDouble;
      mValue.mDouble.SetValue(std::move(aOther.mValue.mDouble.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningDoubleOrString::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or DOMString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or DOMString)");
      return false;
    }
  }
  return true;
}

bool
OwningDoubleOrString::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] double&
OwningDoubleOrString::RawSetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDouble;
  return mValue.mDouble.SetValue();
}

[[nodiscard]] double&
OwningDoubleOrString::SetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  Uninit();
  mType = eDouble;
  return mValue.mDouble.SetValue();
}


void
OwningDoubleOrString::DestroyDouble()
{
  MOZ_RELEASE_ASSERT(IsDouble(), "Wrong type!");
  mValue.mDouble.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningDoubleOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningDoubleOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningDoubleOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningDoubleOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningDoubleOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eDouble: {
      DestroyDouble();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningDoubleOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningDoubleOrString&
OwningDoubleOrString::operator=(OwningDoubleOrString&& aOther)
{
  this->~OwningDoubleOrString();
  new (this) OwningDoubleOrString (std::move(aOther));
  return *this;
}


OwningDoubleOrString&
OwningDoubleOrString::operator=(const OwningDoubleOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      SetAsDouble() = aOther.GetAsDouble();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningDoubleOrSupportedType::OwningDoubleOrSupportedType(OwningDoubleOrSupportedType&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      mType = eDouble;
      mValue.mDouble.SetValue(std::move(aOther.mValue.mDouble.Value()));
      break;
    }
    case eSupportedType: {
      mType = eSupportedType;
      mValue.mSupportedType.SetValue(std::move(aOther.mValue.mSupportedType.Value()));
      break;
    }
  }
}



bool
OwningDoubleOrSupportedType::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or SupportedType)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or SupportedType)");
      return false;
    }
  }
  return true;
}

bool
OwningDoubleOrSupportedType::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] double&
OwningDoubleOrSupportedType::RawSetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDouble;
  return mValue.mDouble.SetValue();
}

[[nodiscard]] double&
OwningDoubleOrSupportedType::SetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  Uninit();
  mType = eDouble;
  return mValue.mDouble.SetValue();
}


void
OwningDoubleOrSupportedType::DestroyDouble()
{
  MOZ_RELEASE_ASSERT(IsDouble(), "Wrong type!");
  mValue.mDouble.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrSupportedType::TrySetToSupportedType(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    SupportedType& memberSlot = RawSetAsSupportedType();
    {
      int index;
      if (!binding_detail::FindEnumStringIndex<true>(cx, value,
                                                                         binding_detail::EnumStrings<SupportedType>::Values,
                                                                         "SupportedType", "SupportedType branch of (double or SupportedType)",
                                                                         &index)) {
        return false;
      }
      MOZ_ASSERT(index >= 0);
      memberSlot = static_cast<SupportedType>(index);
    }
  }
  return true;
}

bool
OwningDoubleOrSupportedType::TrySetToSupportedType(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToSupportedType(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] SupportedType&
OwningDoubleOrSupportedType::RawSetAsSupportedType()
{
  if (mType == eSupportedType) {
    return mValue.mSupportedType.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eSupportedType;
  return mValue.mSupportedType.SetValue();
}

[[nodiscard]] SupportedType&
OwningDoubleOrSupportedType::SetAsSupportedType()
{
  if (mType == eSupportedType) {
    return mValue.mSupportedType.Value();
  }
  Uninit();
  mType = eSupportedType;
  return mValue.mSupportedType.SetValue();
}


void
OwningDoubleOrSupportedType::DestroySupportedType()
{
  MOZ_RELEASE_ASSERT(IsSupportedType(), "Wrong type!");
  mValue.mSupportedType.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrSupportedType::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToSupportedType(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningDoubleOrSupportedType::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningDoubleOrSupportedType::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eDouble: {
      DestroyDouble();
      break;
    }
    case eSupportedType: {
      DestroySupportedType();
      break;
    }
  }
}

bool
OwningDoubleOrSupportedType::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eSupportedType: {
      if (!ToJSValue(cx, mValue.mSupportedType.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningDoubleOrSupportedType&
OwningDoubleOrSupportedType::operator=(OwningDoubleOrSupportedType&& aOther)
{
  this->~OwningDoubleOrSupportedType();
  new (this) OwningDoubleOrSupportedType (std::move(aOther));
  return *this;
}


OwningDoubleOrSupportedType&
OwningDoubleOrSupportedType::operator=(const OwningDoubleOrSupportedType& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      SetAsDouble() = aOther.GetAsDouble();
      break;
    }
    case eSupportedType: {
      SetAsSupportedType() = aOther.GetAsSupportedType();
      break;
    }
  }
  return *this;
}


OwningDoubleOrUSVString::OwningDoubleOrUSVString(OwningDoubleOrUSVString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      mType = eDouble;
      mValue.mDouble.SetValue(std::move(aOther.mValue.mDouble.Value()));
      break;
    }
    case eUSVString: {
      mType = eUSVString;
      mValue.mUSVString.SetValue(std::move(aOther.mValue.mUSVString.Value()));
      break;
    }
  }
}



bool
OwningDoubleOrUSVString::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or USVString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or USVString)");
      return false;
    }
  }
  return true;
}

bool
OwningDoubleOrUSVString::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] double&
OwningDoubleOrUSVString::RawSetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDouble;
  return mValue.mDouble.SetValue();
}

[[nodiscard]] double&
OwningDoubleOrUSVString::SetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  Uninit();
  mType = eDouble;
  return mValue.mDouble.SetValue();
}


void
OwningDoubleOrUSVString::DestroyDouble()
{
  MOZ_RELEASE_ASSERT(IsDouble(), "Wrong type!");
  mValue.mDouble.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrUSVString::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningDoubleOrUSVString::RawSetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}

[[nodiscard]] nsString&
OwningDoubleOrUSVString::SetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  Uninit();
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}



void
OwningDoubleOrUSVString::DestroyUSVString()
{
  MOZ_RELEASE_ASSERT(IsUSVString(), "Wrong type!");
  mValue.mUSVString.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrUSVString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningDoubleOrUSVString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningDoubleOrUSVString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eDouble: {
      DestroyDouble();
      break;
    }
    case eUSVString: {
      DestroyUSVString();
      break;
    }
  }
}

bool
OwningDoubleOrUSVString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningDoubleOrUSVString&
OwningDoubleOrUSVString::operator=(OwningDoubleOrUSVString&& aOther)
{
  this->~OwningDoubleOrUSVString();
  new (this) OwningDoubleOrUSVString (std::move(aOther));
  return *this;
}


OwningDoubleOrUSVString&
OwningDoubleOrUSVString::operator=(const OwningDoubleOrUSVString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      SetAsDouble() = aOther.GetAsDouble();
      break;
    }
    case eUSVString: {
      SetAsUSVString() = aOther.GetAsUSVString();
      break;
    }
  }
  return *this;
}


OwningDoubleOrUTF8String::OwningDoubleOrUTF8String(OwningDoubleOrUTF8String&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      mType = eDouble;
      mValue.mDouble.SetValue(std::move(aOther.mValue.mDouble.Value()));
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
  }
}



bool
OwningDoubleOrUTF8String::TrySetToDouble(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Double branch of (double or USVString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Double branch of (double or USVString)");
      return false;
    }
  }
  return true;
}

bool
OwningDoubleOrUTF8String::TrySetToDouble(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDouble(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] double&
OwningDoubleOrUTF8String::RawSetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDouble;
  return mValue.mDouble.SetValue();
}

[[nodiscard]] double&
OwningDoubleOrUTF8String::SetAsDouble()
{
  if (mType == eDouble) {
    return mValue.mDouble.Value();
  }
  Uninit();
  mType = eDouble;
  return mValue.mDouble.SetValue();
}


void
OwningDoubleOrUTF8String::DestroyDouble()
{
  MOZ_RELEASE_ASSERT(IsDouble(), "Wrong type!");
  mValue.mDouble.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrUTF8String::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningDoubleOrUTF8String::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningDoubleOrUTF8String::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningDoubleOrUTF8String::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningDoubleOrUTF8String::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningDoubleOrUTF8String::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningDoubleOrUTF8String::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eDouble: {
      DestroyDouble();
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
  }
}

bool
OwningDoubleOrUTF8String::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eDouble: {
      rval.set(JS_NumberValue(double(mValue.mDouble.Value())));
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningDoubleOrUTF8String&
OwningDoubleOrUTF8String::operator=(OwningDoubleOrUTF8String&& aOther)
{
  this->~OwningDoubleOrUTF8String();
  new (this) OwningDoubleOrUTF8String (std::move(aOther));
  return *this;
}


OwningDoubleOrUTF8String&
OwningDoubleOrUTF8String::operator=(const OwningDoubleOrUTF8String& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eDouble: {
      SetAsDouble() = aOther.GetAsDouble();
      break;
    }
    case eUTF8String: {
      SetAsUTF8String() = aOther.GetAsUTF8String();
      break;
    }
  }
  return *this;
}


OwningFileOrDirectory::OwningFileOrDirectory(OwningFileOrDirectory&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      mType = eFile;
      mValue.mFile.SetValue(std::move(aOther.mValue.mFile.Value()));
      break;
    }
    case eDirectory: {
      mType = eDirectory;
      mValue.mDirectory.SetValue(std::move(aOther.mValue.mDirectory.Value()));
      break;
    }
  }
}



bool
OwningFileOrDirectory::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    static_assert(IsRefcounted<mozilla::dom::File>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrDirectory::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrDirectory::RawSetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFile;
  return mValue.mFile.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrDirectory::SetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  Uninit();
  mType = eFile;
  return mValue.mFile.SetValue();
}


void
OwningFileOrDirectory::DestroyFile()
{
  MOZ_RELEASE_ASSERT(IsFile(), "Wrong type!");
  mValue.mFile.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrDirectory::TrySetToDirectory(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::Directory>& memberSlot = RawSetAsDirectory();
    static_assert(IsRefcounted<mozilla::dom::Directory>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Directory, mozilla::dom::Directory>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyDirectory();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrDirectory::TrySetToDirectory(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDirectory(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::Directory>&
OwningFileOrDirectory::RawSetAsDirectory()
{
  if (mType == eDirectory) {
    return mValue.mDirectory.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDirectory;
  return mValue.mDirectory.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::Directory>&
OwningFileOrDirectory::SetAsDirectory()
{
  if (mType == eDirectory) {
    return mValue.mDirectory.Value();
  }
  Uninit();
  mType = eDirectory;
  return mValue.mDirectory.SetValue();
}


void
OwningFileOrDirectory::DestroyDirectory()
{
  MOZ_RELEASE_ASSERT(IsDirectory(), "Wrong type!");
  mValue.mDirectory.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrDirectory::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToDirectory(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, Directory");
    return false;
  }
  return true;
}

bool
OwningFileOrDirectory::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningFileOrDirectory::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eFile: {
      DestroyFile();
      break;
    }
    case eDirectory: {
      DestroyDirectory();
      break;
    }
  }
}

bool
OwningFileOrDirectory::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eDirectory: {
      if (!GetOrCreateDOMReflector(cx, mValue.mDirectory.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningFileOrDirectory&
OwningFileOrDirectory::operator=(OwningFileOrDirectory&& aOther)
{
  this->~OwningFileOrDirectory();
  new (this) OwningFileOrDirectory (std::move(aOther));
  return *this;
}


OwningFileOrDirectory&
OwningFileOrDirectory::operator=(const OwningFileOrDirectory& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      SetAsFile() = aOther.GetAsFile();
      break;
    }
    case eDirectory: {
      SetAsDirectory() = aOther.GetAsDirectory();
      break;
    }
  }
  return *this;
}


OwningFileOrUSVStringOrFormData::OwningFileOrUSVStringOrFormData(OwningFileOrUSVStringOrFormData&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      mType = eFile;
      mValue.mFile.SetValue(std::move(aOther.mValue.mFile.Value()));
      break;
    }
    case eUSVString: {
      mType = eUSVString;
      mValue.mUSVString.SetValue(std::move(aOther.mValue.mUSVString.Value()));
      break;
    }
    case eFormData: {
      mType = eFormData;
      mValue.mFormData.SetValue(std::move(aOther.mValue.mFormData.Value()));
      break;
    }
  }
}



bool
OwningFileOrUSVStringOrFormData::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    static_assert(IsRefcounted<mozilla::dom::File>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrUSVStringOrFormData::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrUSVStringOrFormData::RawSetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFile;
  return mValue.mFile.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrUSVStringOrFormData::SetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  Uninit();
  mType = eFile;
  return mValue.mFile.SetValue();
}


void
OwningFileOrUSVStringOrFormData::DestroyFile()
{
  MOZ_RELEASE_ASSERT(IsFile(), "Wrong type!");
  mValue.mFile.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrUSVStringOrFormData::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningFileOrUSVStringOrFormData::RawSetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}

[[nodiscard]] nsString&
OwningFileOrUSVStringOrFormData::SetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  Uninit();
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}



void
OwningFileOrUSVStringOrFormData::DestroyUSVString()
{
  MOZ_RELEASE_ASSERT(IsUSVString(), "Wrong type!");
  mValue.mUSVString.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrUSVStringOrFormData::TrySetToFormData(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::FormData>& memberSlot = RawSetAsFormData();
    static_assert(IsRefcounted<mozilla::dom::FormData>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::FormData, mozilla::dom::FormData>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFormData();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrUSVStringOrFormData::TrySetToFormData(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFormData(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::FormData>&
OwningFileOrUSVStringOrFormData::RawSetAsFormData()
{
  if (mType == eFormData) {
    return mValue.mFormData.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFormData;
  return mValue.mFormData.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::FormData>&
OwningFileOrUSVStringOrFormData::SetAsFormData()
{
  if (mType == eFormData) {
    return mValue.mFormData.Value();
  }
  Uninit();
  mType = eFormData;
  return mValue.mFormData.SetValue();
}


void
OwningFileOrUSVStringOrFormData::DestroyFormData()
{
  MOZ_RELEASE_ASSERT(IsFormData(), "Wrong type!");
  mValue.mFormData.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrUSVStringOrFormData::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToFormData(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, FormData");
    return false;
  }
  return true;
}

bool
OwningFileOrUSVStringOrFormData::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningFileOrUSVStringOrFormData::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eFile: {
      DestroyFile();
      break;
    }
    case eUSVString: {
      DestroyUSVString();
      break;
    }
    case eFormData: {
      DestroyFormData();
      break;
    }
  }
}

bool
OwningFileOrUSVStringOrFormData::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eFormData: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFormData.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningFileOrUSVStringOrFormData&
OwningFileOrUSVStringOrFormData::operator=(OwningFileOrUSVStringOrFormData&& aOther)
{
  this->~OwningFileOrUSVStringOrFormData();
  new (this) OwningFileOrUSVStringOrFormData (std::move(aOther));
  return *this;
}


OwningFileOrUSVStringOrFormData&
OwningFileOrUSVStringOrFormData::operator=(const OwningFileOrUSVStringOrFormData& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      SetAsFile() = aOther.GetAsFile();
      break;
    }
    case eUSVString: {
      SetAsUSVString() = aOther.GetAsUSVString();
      break;
    }
    case eFormData: {
      SetAsFormData() = aOther.GetAsFormData();
      break;
    }
  }
  return *this;
}


OwningFloatOrString::OwningFloatOrString(OwningFloatOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFloat: {
      mType = eFloat;
      mValue.mFloat.SetValue(std::move(aOther.mValue.mFloat.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningFloatOrString::TrySetToFloat(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    float& memberSlot = RawSetAsFloat();
    if (!ValueToPrimitive<float, eDefault>(cx, value, "Float branch of (float or DOMString)", &memberSlot)) {
      return false;
    } else if (!std::isfinite(memberSlot)) {
      cx.ThrowErrorMessage<MSG_NOT_FINITE>("Float branch of (float or DOMString)");
      return false;
    }
  }
  return true;
}

bool
OwningFloatOrString::TrySetToFloat(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFloat(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] float&
OwningFloatOrString::RawSetAsFloat()
{
  if (mType == eFloat) {
    return mValue.mFloat.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFloat;
  return mValue.mFloat.SetValue();
}

[[nodiscard]] float&
OwningFloatOrString::SetAsFloat()
{
  if (mType == eFloat) {
    return mValue.mFloat.Value();
  }
  Uninit();
  mType = eFloat;
  return mValue.mFloat.SetValue();
}


void
OwningFloatOrString::DestroyFloat()
{
  MOZ_RELEASE_ASSERT(IsFloat(), "Wrong type!");
  mValue.mFloat.Destroy();
  mType = eUninitialized;
}



bool
OwningFloatOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningFloatOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningFloatOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningFloatOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningFloatOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToFloat(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningFloatOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningFloatOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eFloat: {
      DestroyFloat();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningFloatOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFloat: {
      rval.set(JS_NumberValue(double(mValue.mFloat.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningFloatOrString&
OwningFloatOrString::operator=(OwningFloatOrString&& aOther)
{
  this->~OwningFloatOrString();
  new (this) OwningFloatOrString (std::move(aOther));
  return *this;
}


OwningFloatOrString&
OwningFloatOrString::operator=(const OwningFloatOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFloat: {
      SetAsFloat() = aOther.GetAsFloat();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningHTMLCanvasElementOrOffscreenCanvas::OwningHTMLCanvasElementOrOffscreenCanvas(OwningHTMLCanvasElementOrOffscreenCanvas&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLCanvasElement: {
      mType = eHTMLCanvasElement;
      mValue.mHTMLCanvasElement.SetValue(std::move(aOther.mValue.mHTMLCanvasElement.Value()));
      break;
    }
    case eOffscreenCanvas: {
      mType = eOffscreenCanvas;
      mValue.mOffscreenCanvas.SetValue(std::move(aOther.mValue.mOffscreenCanvas.Value()));
      break;
    }
  }
}



bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::HTMLCanvasElement>& memberSlot = RawSetAsHTMLCanvasElement();
    static_assert(IsRefcounted<mozilla::dom::HTMLCanvasElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLCanvasElement, mozilla::dom::HTMLCanvasElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLCanvasElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLCanvasElement>&
OwningHTMLCanvasElementOrOffscreenCanvas::RawSetAsHTMLCanvasElement()
{
  if (mType == eHTMLCanvasElement) {
    return mValue.mHTMLCanvasElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLCanvasElement;
  return mValue.mHTMLCanvasElement.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLCanvasElement>&
OwningHTMLCanvasElementOrOffscreenCanvas::SetAsHTMLCanvasElement()
{
  if (mType == eHTMLCanvasElement) {
    return mValue.mHTMLCanvasElement.Value();
  }
  Uninit();
  mType = eHTMLCanvasElement;
  return mValue.mHTMLCanvasElement.SetValue();
}


void
OwningHTMLCanvasElementOrOffscreenCanvas::DestroyHTMLCanvasElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLCanvasElement(), "Wrong type!");
  mValue.mHTMLCanvasElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::OffscreenCanvas>& memberSlot = RawSetAsOffscreenCanvas();
    static_assert(IsRefcounted<mozilla::dom::OffscreenCanvas>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::OffscreenCanvas, mozilla::dom::OffscreenCanvas>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyOffscreenCanvas();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::OffscreenCanvas>&
OwningHTMLCanvasElementOrOffscreenCanvas::RawSetAsOffscreenCanvas()
{
  if (mType == eOffscreenCanvas) {
    return mValue.mOffscreenCanvas.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eOffscreenCanvas;
  return mValue.mOffscreenCanvas.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::OffscreenCanvas>&
OwningHTMLCanvasElementOrOffscreenCanvas::SetAsOffscreenCanvas()
{
  if (mType == eOffscreenCanvas) {
    return mValue.mOffscreenCanvas.Value();
  }
  Uninit();
  mType = eOffscreenCanvas;
  return mValue.mOffscreenCanvas.SetValue();
}


void
OwningHTMLCanvasElementOrOffscreenCanvas::DestroyOffscreenCanvas()
{
  MOZ_RELEASE_ASSERT(IsOffscreenCanvas(), "Wrong type!");
  mValue.mOffscreenCanvas.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLCanvasElementOrOffscreenCanvas::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLCanvasElement, OffscreenCanvas");
    return false;
  }
  return true;
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningHTMLCanvasElementOrOffscreenCanvas::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eHTMLCanvasElement: {
      DestroyHTMLCanvasElement();
      break;
    }
    case eOffscreenCanvas: {
      DestroyOffscreenCanvas();
      break;
    }
  }
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLCanvasElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLCanvasElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eOffscreenCanvas: {
      if (!GetOrCreateDOMReflector(cx, mValue.mOffscreenCanvas.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningHTMLCanvasElementOrOffscreenCanvas&
OwningHTMLCanvasElementOrOffscreenCanvas::operator=(OwningHTMLCanvasElementOrOffscreenCanvas&& aOther)
{
  this->~OwningHTMLCanvasElementOrOffscreenCanvas();
  new (this) OwningHTMLCanvasElementOrOffscreenCanvas (std::move(aOther));
  return *this;
}


OwningHTMLCanvasElementOrOffscreenCanvas&
OwningHTMLCanvasElementOrOffscreenCanvas::operator=(const OwningHTMLCanvasElementOrOffscreenCanvas& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLCanvasElement: {
      SetAsHTMLCanvasElement() = aOther.GetAsHTMLCanvasElement();
      break;
    }
    case eOffscreenCanvas: {
      SetAsOffscreenCanvas() = aOther.GetAsOffscreenCanvas();
      break;
    }
  }
  return *this;
}


OwningHTMLElementOrLong::OwningHTMLElementOrLong(OwningHTMLElementOrLong&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLElement: {
      mType = eHTMLElement;
      mValue.mHTMLElement.SetValue(std::move(aOther.mValue.mHTMLElement.Value()));
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
  }
}



bool
OwningHTMLElementOrLong::TrySetToHTMLElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<nsGenericHTMLElement>& memberSlot = RawSetAsHTMLElement();
    static_assert(IsRefcounted<nsGenericHTMLElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLElement, nsGenericHTMLElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLElementOrLong::TrySetToHTMLElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<nsGenericHTMLElement>&
OwningHTMLElementOrLong::RawSetAsHTMLElement()
{
  if (mType == eHTMLElement) {
    return mValue.mHTMLElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLElement;
  return mValue.mHTMLElement.SetValue();
}

[[nodiscard]] OwningNonNull<nsGenericHTMLElement>&
OwningHTMLElementOrLong::SetAsHTMLElement()
{
  if (mType == eHTMLElement) {
    return mValue.mHTMLElement.Value();
  }
  Uninit();
  mType = eHTMLElement;
  return mValue.mHTMLElement.SetValue();
}


void
OwningHTMLElementOrLong::DestroyHTMLElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLElement(), "Wrong type!");
  mValue.mHTMLElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLElementOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (HTMLElement or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningHTMLElementOrLong::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningHTMLElementOrLong::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningHTMLElementOrLong::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLElementOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLElement");
    return false;
  }
  return true;
}

bool
OwningHTMLElementOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningHTMLElementOrLong::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eHTMLElement: {
      DestroyHTMLElement();
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
  }
}

bool
OwningHTMLElementOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningHTMLElementOrLong&
OwningHTMLElementOrLong::operator=(OwningHTMLElementOrLong&& aOther)
{
  this->~OwningHTMLElementOrLong();
  new (this) OwningHTMLElementOrLong (std::move(aOther));
  return *this;
}


OwningHTMLElementOrLong&
OwningHTMLElementOrLong::operator=(const OwningHTMLElementOrLong& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLElement: {
      SetAsHTMLElement() = aOther.GetAsHTMLElement();
      break;
    }
    case eLong: {
      SetAsLong() = aOther.GetAsLong();
      break;
    }
  }
  return *this;
}


OwningHTMLOptionElementOrHTMLOptGroupElement::OwningHTMLOptionElementOrHTMLOptGroupElement(OwningHTMLOptionElementOrHTMLOptGroupElement&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLOptionElement: {
      mType = eHTMLOptionElement;
      mValue.mHTMLOptionElement.SetValue(std::move(aOther.mValue.mHTMLOptionElement.Value()));
      break;
    }
    case eHTMLOptGroupElement: {
      mType = eHTMLOptGroupElement;
      mValue.mHTMLOptGroupElement.SetValue(std::move(aOther.mValue.mHTMLOptGroupElement.Value()));
      break;
    }
  }
}



bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::HTMLOptionElement>& memberSlot = RawSetAsHTMLOptionElement();
    static_assert(IsRefcounted<mozilla::dom::HTMLOptionElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptionElement, mozilla::dom::HTMLOptionElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptionElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptionElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::RawSetAsHTMLOptionElement()
{
  if (mType == eHTMLOptionElement) {
    return mValue.mHTMLOptionElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLOptionElement;
  return mValue.mHTMLOptionElement.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptionElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::SetAsHTMLOptionElement()
{
  if (mType == eHTMLOptionElement) {
    return mValue.mHTMLOptionElement.Value();
  }
  Uninit();
  mType = eHTMLOptionElement;
  return mValue.mHTMLOptionElement.SetValue();
}


void
OwningHTMLOptionElementOrHTMLOptGroupElement::DestroyHTMLOptionElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLOptionElement(), "Wrong type!");
  mValue.mHTMLOptionElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::HTMLOptGroupElement>& memberSlot = RawSetAsHTMLOptGroupElement();
    static_assert(IsRefcounted<mozilla::dom::HTMLOptGroupElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptGroupElement, mozilla::dom::HTMLOptGroupElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptGroupElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptGroupElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::RawSetAsHTMLOptGroupElement()
{
  if (mType == eHTMLOptGroupElement) {
    return mValue.mHTMLOptGroupElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLOptGroupElement;
  return mValue.mHTMLOptGroupElement.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptGroupElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::SetAsHTMLOptGroupElement()
{
  if (mType == eHTMLOptGroupElement) {
    return mValue.mHTMLOptGroupElement.Value();
  }
  Uninit();
  mType = eHTMLOptGroupElement;
  return mValue.mHTMLOptGroupElement.SetValue();
}


void
OwningHTMLOptionElementOrHTMLOptGroupElement::DestroyHTMLOptGroupElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLOptGroupElement(), "Wrong type!");
  mValue.mHTMLOptGroupElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLOptionElementOrHTMLOptGroupElement::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLOptionElement, HTMLOptGroupElement");
    return false;
  }
  return true;
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningHTMLOptionElementOrHTMLOptGroupElement::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eHTMLOptionElement: {
      DestroyHTMLOptionElement();
      break;
    }
    case eHTMLOptGroupElement: {
      DestroyHTMLOptGroupElement();
      break;
    }
  }
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLOptionElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptionElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eHTMLOptGroupElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptGroupElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningHTMLOptionElementOrHTMLOptGroupElement&
OwningHTMLOptionElementOrHTMLOptGroupElement::operator=(OwningHTMLOptionElementOrHTMLOptGroupElement&& aOther)
{
  this->~OwningHTMLOptionElementOrHTMLOptGroupElement();
  new (this) OwningHTMLOptionElementOrHTMLOptGroupElement (std::move(aOther));
  return *this;
}


OwningHTMLOptionElementOrHTMLOptGroupElement&
OwningHTMLOptionElementOrHTMLOptGroupElement::operator=(const OwningHTMLOptionElementOrHTMLOptGroupElement& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLOptionElement: {
      SetAsHTMLOptionElement() = aOther.GetAsHTMLOptionElement();
      break;
    }
    case eHTMLOptGroupElement: {
      SetAsHTMLOptGroupElement() = aOther.GetAsHTMLOptGroupElement();
      break;
    }
  }
  return *this;
}


OwningLongOrStringAnyRecord::OwningLongOrStringAnyRecord(OwningLongOrStringAnyRecord&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
    case eStringAnyRecord: {
      mType = eStringAnyRecord;
      mValue.mStringAnyRecord.SetValue(std::move(aOther.mValue.mStringAnyRecord.Value()));
      break;
    }
  }
}


bool
OwningLongOrStringAnyRecord::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (long or record<DOMString, any>)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningLongOrStringAnyRecord::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningLongOrStringAnyRecord::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningLongOrStringAnyRecord::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningLongOrStringAnyRecord::TrySetToStringAnyRecord(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Record<nsString, JS::Value>& memberSlot = RawSetAsStringAnyRecord();
    auto& recordEntries = memberSlot.Entries();

    JS::Rooted<JSObject*> recordObj(cx, &value.toObject());
    JS::RootedVector<jsid> ids(cx);
    if (!js::GetPropertyKeys(cx, recordObj,
                             JSITER_OWNONLY | JSITER_HIDDEN | JSITER_SYMBOLS, &ids)) {
      return false;
    }
    if (!recordEntries.SetCapacity(ids.length(), mozilla::fallible)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
    JS::Rooted<JS::Value> propNameValue(cx);
    JS::Rooted<JS::Value> temp(cx);
    JS::Rooted<jsid> curId(cx);
    JS::Rooted<JS::Value> idVal(cx);
    // Use a hashset to keep track of ids seen, to avoid
    // introducing nasty O(N^2) behavior scanning for them all the
    // time.  Ideally we'd use a data structure with O(1) lookup
    // _and_ ordering for the MozMap, but we don't have one lying
    // around.
    nsTHashtable<nsStringHashKey> idsSeen;
    for (size_t i = 0; i < ids.length(); ++i) {
      curId = ids[i];

      JS::Rooted<mozilla::Maybe<JS::PropertyDescriptor>> desc(cx);
      if (!JS_GetOwnPropertyDescriptorById(cx, recordObj, curId,
                                           &desc)) {
        return false;
      }

      if (desc.isNothing() || !desc->enumerable()) {
        continue;
      }

      idVal = js::IdToValue(curId);
      nsString propName;
      // This will just throw if idVal is a Symbol, like the spec says
      // to do.
      if (!ConvertJSValueToString(cx, idVal, "key of record<DOMString, any> branch of (long or record<DOMString, any>)", propName)) {
        return false;
      }

      if (!JS_GetPropertyById(cx, recordObj, curId, &temp)) {
        return false;
      }

      Record<nsString, JS::Value>::EntryType* entry;
      if (!idsSeen.EnsureInserted(propName)) {
        // Find the existing entry.
        auto idx = recordEntries.IndexOf(propName);
        MOZ_ASSERT(idx != recordEntries.NoIndex,
                   "Why is it not found?");
        // Now blow it away to make it look like it was just added
        // to the array, because it's not obvious that it's
        // safe to write to its already-initialized mValue via our
        // normal codegen conversions.  For example, the value
        // could be a union and this would change its type, but
        // codegen assumes we won't do that.
        entry = recordEntries.ReconstructElementAt(idx);
      } else {
        // Safe to do an infallible append here, because we did a
        // SetCapacity above to the right capacity.
        entry = recordEntries.AppendElement();
      }
      entry->mKey = propName;
      JS::Value& slot = entry->mValue;
#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunreachable-code"
#pragma clang diagnostic ignored "-Wunreachable-code-return"
#endif // __clang__
      if ((passedToJSImpl) && !CallerSubsumes(temp)) {
        cx.ThrowErrorMessage<MSG_PERMISSION_DENIED_TO_PASS_ARG>("value in record<DOMString, any> branch of (long or record<DOMString, any>)");
        return false;
      }
#ifdef __clang__
#pragma clang diagnostic pop
#endif // __clang__
      slot = temp;
    }
  }
  return true;
}

bool
OwningLongOrStringAnyRecord::TrySetToStringAnyRecord(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToStringAnyRecord(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Record<nsString, JS::Value>&
OwningLongOrStringAnyRecord::RawSetAsStringAnyRecord()
{
  if (mType == eStringAnyRecord) {
    return mValue.mStringAnyRecord.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eStringAnyRecord;
  return mValue.mStringAnyRecord.SetValue();
}

[[nodiscard]] Record<nsString, JS::Value>&
OwningLongOrStringAnyRecord::SetAsStringAnyRecord()
{
  if (mType == eStringAnyRecord) {
    return mValue.mStringAnyRecord.Value();
  }
  Uninit();
  mType = eStringAnyRecord;
  return mValue.mStringAnyRecord.SetValue();
}


void
OwningLongOrStringAnyRecord::DestroyStringAnyRecord()
{
  MOZ_RELEASE_ASSERT(IsStringAnyRecord(), "Wrong type!");
  mValue.mStringAnyRecord.Destroy();
  mType = eUninitialized;
}



bool
OwningLongOrStringAnyRecord::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!done) {
      done = (failed = !TrySetToStringAnyRecord(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
  }
  if (!done) {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "record<DOMString, any>");
    return false;
  }
  return true;
}

bool
OwningLongOrStringAnyRecord::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningLongOrStringAnyRecord::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
    case eStringAnyRecord: {
      DestroyStringAnyRecord();
      break;
    }
  }
}

bool
OwningLongOrStringAnyRecord::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    case eStringAnyRecord: {

      JS::Rooted<JSObject*> returnObj(cx, JS_NewPlainObject(cx));
      if (!returnObj) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (auto& entry : mValue.mStringAnyRecord.Value().Entries()) {
          auto& recordValue0 = entry.mValue;
          // Control block to let us common up the JS_DefineUCProperty calls when there
          // are different ways to succeed at wrapping the value.
          do {
            JS::ExposeValueToActiveJS(recordValue0);
            tmp.set(recordValue0);
            if (!MaybeWrapValue(cx, &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineUCProperty(cx, returnObj,
                                   entry.mKey.BeginReading(),
                                   entry.mKey.Length(), tmp,
                                   JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnObj);
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningLongOrStringAnyRecord::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eStringAnyRecord: {
      TraceRecord(trc, mValue.mStringAnyRecord.Value());
      break;
    }
    default: {
    }
  }
}

OwningLongOrStringAnyRecord&
OwningLongOrStringAnyRecord::operator=(OwningLongOrStringAnyRecord&& aOther)
{
  this->~OwningLongOrStringAnyRecord();
  new (this) OwningLongOrStringAnyRecord (std::move(aOther));
  return *this;
}



OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array(OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eInt8Array: {
      mType = eInt8Array;
      mValue.mInt8Array.SetValue(std::move(aOther.mValue.mInt8Array.Value()));
      break;
    }
    case eInt16Array: {
      mType = eInt16Array;
      mValue.mInt16Array.SetValue(std::move(aOther.mValue.mInt16Array.Value()));
      break;
    }
  }
}


bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt8Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Int8Array& memberSlot = RawSetAsInt8Array();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyInt8Array();
      tryNext = true;
      return true;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Int8Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Int8Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Int8Array branch of (Int8Array or Int16Array)");
      return false;
    }
  }
  return true;
}

bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt8Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToInt8Array(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Int8Array&
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::RawSetAsInt8Array()
{
  if (mType == eInt8Array) {
    return mValue.mInt8Array.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eInt8Array;
  return mValue.mInt8Array.SetValue();
}

[[nodiscard]] Int8Array&
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::SetAsInt8Array()
{
  if (mType == eInt8Array) {
    return mValue.mInt8Array.Value();
  }
  Uninit();
  mType = eInt8Array;
  return mValue.mInt8Array.SetValue();
}


void
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::DestroyInt8Array()
{
  MOZ_RELEASE_ASSERT(IsInt8Array(), "Wrong type!");
  mValue.mInt8Array.Destroy();
  mType = eUninitialized;
}



bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt16Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Int16Array& memberSlot = RawSetAsInt16Array();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyInt16Array();
      tryNext = true;
      return true;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Int16Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Int16Array branch of (Int8Array or Int16Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Int16Array branch of (Int8Array or Int16Array)");
      return false;
    }
  }
  return true;
}

bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::TrySetToInt16Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToInt16Array(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Int16Array&
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::RawSetAsInt16Array()
{
  if (mType == eInt16Array) {
    return mValue.mInt16Array.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eInt16Array;
  return mValue.mInt16Array.SetValue();
}

[[nodiscard]] Int16Array&
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::SetAsInt16Array()
{
  if (mType == eInt16Array) {
    return mValue.mInt16Array.Value();
  }
  Uninit();
  mType = eInt16Array;
  return mValue.mInt16Array.SetValue();
}


void
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::DestroyInt16Array()
{
  MOZ_RELEASE_ASSERT(IsInt16Array(), "Wrong type!");
  mValue.mInt16Array.Destroy();
  mType = eUninitialized;
}



bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToInt8Array(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToInt16Array(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Int8Array, Int16Array");
    return false;
  }
  return true;
}

bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eInt8Array: {
      DestroyInt8Array();
      break;
    }
    case eInt16Array: {
      DestroyInt16Array();
      break;
    }
  }
}

bool
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eInt8Array: {
      rval.setObject(*mValue.mInt8Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eInt16Array: {
      rval.setObject(*mValue.mInt16Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eInt8Array: {
      mValue.mInt8Array.Value().TraceSelf(trc);
      break;
    }
    case eInt16Array: {
      mValue.mInt16Array.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array&
OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array::operator=(OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array&& aOther)
{
  this->~OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array();
  new (this) OwningMaybeSharedInt8ArrayOrMaybeSharedInt16Array (std::move(aOther));
  return *this;
}



OwningNodeOrString::OwningNodeOrString(OwningNodeOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNode: {
      mType = eNode;
      mValue.mNode.SetValue(std::move(aOther.mValue.mNode.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningNodeOrString::TrySetToNode(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<nsINode>& memberSlot = RawSetAsNode();
    static_assert(IsRefcounted<nsINode>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Node, nsINode>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyNode();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningNodeOrString::TrySetToNode(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToNode(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<nsINode>&
OwningNodeOrString::RawSetAsNode()
{
  if (mType == eNode) {
    return mValue.mNode.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eNode;
  return mValue.mNode.SetValue();
}

[[nodiscard]] OwningNonNull<nsINode>&
OwningNodeOrString::SetAsNode()
{
  if (mType == eNode) {
    return mValue.mNode.Value();
  }
  Uninit();
  mType = eNode;
  return mValue.mNode.SetValue();
}


void
OwningNodeOrString::DestroyNode()
{
  MOZ_RELEASE_ASSERT(IsNode(), "Wrong type!");
  mValue.mNode.Destroy();
  mType = eUninitialized;
}



bool
OwningNodeOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningNodeOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningNodeOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningNodeOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningNodeOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToNode(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Node");
    return false;
  }
  return true;
}

bool
OwningNodeOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningNodeOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNode: {
      DestroyNode();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningNodeOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNode: {
      if (!GetOrCreateDOMReflector(cx, mValue.mNode.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningNodeOrString&
OwningNodeOrString::operator=(OwningNodeOrString&& aOther)
{
  this->~OwningNodeOrString();
  new (this) OwningNodeOrString (std::move(aOther));
  return *this;
}


OwningNodeOrString&
OwningNodeOrString::operator=(const OwningNodeOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNode: {
      SetAsNode() = aOther.GetAsNode();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningObjectOrLong::OwningObjectOrLong(OwningObjectOrLong&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eObject: {
      mType = eObject;
      mValue.mObject.SetValue(std::move(aOther.mValue.mObject.Value()));
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
  }
}



[[nodiscard]] JSObject*&
OwningObjectOrLong::RawSetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eObject;
  return mValue.mObject.SetValue();
}

[[nodiscard]] JSObject*&
OwningObjectOrLong::SetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  Uninit();
  mType = eObject;
  return mValue.mObject.SetValue();
}


void
OwningObjectOrLong::DestroyObject()
{
  MOZ_RELEASE_ASSERT(IsObject(), "Wrong type!");
  mValue.mObject.Destroy();
  mType = eUninitialized;
}



bool
OwningObjectOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (object or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningObjectOrLong::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningObjectOrLong::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningObjectOrLong::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningObjectOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
OwningObjectOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningObjectOrLong::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eObject: {
      DestroyObject();
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
  }
}

bool
OwningObjectOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningObjectOrLong::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eObject: {
      JS::TraceRoot(trc, &mValue.mObject.Value(), "mValue.mObject");
      break;
    }
    default: {
    }
  }
}

OwningObjectOrLong&
OwningObjectOrLong::operator=(OwningObjectOrLong&& aOther)
{
  this->~OwningObjectOrLong();
  new (this) OwningObjectOrLong (std::move(aOther));
  return *this;
}



OwningObjectOrNullOrLong::OwningObjectOrNullOrLong(OwningObjectOrNullOrLong&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eObject: {
      mType = eObject;
      mValue.mObject.SetValue(std::move(aOther.mValue.mObject.Value()));
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
  }
}





[[nodiscard]] JSObject*&
OwningObjectOrNullOrLong::RawSetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eObject;
  return mValue.mObject.SetValue();
}

[[nodiscard]] JSObject*&
OwningObjectOrNullOrLong::SetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  Uninit();
  mType = eObject;
  return mValue.mObject.SetValue();
}


void
OwningObjectOrNullOrLong::DestroyObject()
{
  MOZ_RELEASE_ASSERT(IsObject(), "Wrong type!");
  mValue.mObject.Destroy();
  mType = eUninitialized;
}



bool
OwningObjectOrNullOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (object? or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningObjectOrNullOrLong::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningObjectOrNullOrLong::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningObjectOrNullOrLong::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningObjectOrNullOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
        return false;
      }
      done = true;
    } else {
      do {
        done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
        break;
      } while (false);
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
      return false;
    }
  }
  return true;
}

bool
OwningObjectOrNullOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningObjectOrNullOrLong::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNull: {
      break;
    }
    case eObject: {
      DestroyObject();
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
  }
}

bool
OwningObjectOrNullOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningObjectOrNullOrLong::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eObject: {
      JS::TraceRoot(trc, &mValue.mObject.Value(), "mValue.mObject");
      break;
    }
    default: {
    }
  }
}

OwningObjectOrNullOrLong&
OwningObjectOrNullOrLong::operator=(OwningObjectOrNullOrLong&& aOther)
{
  this->~OwningObjectOrNullOrLong();
  new (this) OwningObjectOrNullOrLong (std::move(aOther));
  return *this;
}



OwningStringOrArrayBuffer::OwningStringOrArrayBuffer(OwningStringOrArrayBuffer&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
  }
}


bool
OwningStringOrArrayBuffer::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrArrayBuffer::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrArrayBuffer::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrArrayBuffer::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
OwningStringOrArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningStringOrArrayBuffer::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningStringOrArrayBuffer::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningStringOrArrayBuffer::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
    return false;
  }
  return true;
}

bool
OwningStringOrArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrArrayBuffer::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
  }
}

bool
OwningStringOrArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningStringOrArrayBuffer::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningStringOrArrayBuffer&
OwningStringOrArrayBuffer::operator=(OwningStringOrArrayBuffer&& aOther)
{
  this->~OwningStringOrArrayBuffer();
  new (this) OwningStringOrArrayBuffer (std::move(aOther));
  return *this;
}



OwningStringOrBooleanOrObject::OwningStringOrBooleanOrObject(OwningStringOrBooleanOrObject&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eBoolean: {
      mType = eBoolean;
      mValue.mBoolean.SetValue(std::move(aOther.mValue.mBoolean.Value()));
      break;
    }
    case eObject: {
      mType = eObject;
      mValue.mObject.SetValue(std::move(aOther.mValue.mObject.Value()));
      break;
    }
  }
}


bool
OwningStringOrBooleanOrObject::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrBooleanOrObject::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrBooleanOrObject::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrBooleanOrObject::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrBooleanOrObject::TrySetToBoolean(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    bool& memberSlot = RawSetAsBoolean();
    if (!ValueToPrimitive<bool, eDefault>(cx, value, "Boolean branch of (DOMString or boolean or object)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] bool&
OwningStringOrBooleanOrObject::RawSetAsBoolean()
{
  if (mType == eBoolean) {
    return mValue.mBoolean.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eBoolean;
  return mValue.mBoolean.SetValue();
}

[[nodiscard]] bool&
OwningStringOrBooleanOrObject::SetAsBoolean()
{
  if (mType == eBoolean) {
    return mValue.mBoolean.Value();
  }
  Uninit();
  mType = eBoolean;
  return mValue.mBoolean.SetValue();
}


void
OwningStringOrBooleanOrObject::DestroyBoolean()
{
  MOZ_RELEASE_ASSERT(IsBoolean(), "Wrong type!");
  mValue.mBoolean.Destroy();
  mType = eUninitialized;
}




[[nodiscard]] JSObject*&
OwningStringOrBooleanOrObject::RawSetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eObject;
  return mValue.mObject.SetValue();
}

[[nodiscard]] JSObject*&
OwningStringOrBooleanOrObject::SetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  Uninit();
  mType = eObject;
  return mValue.mObject.SetValue();
}


void
OwningStringOrBooleanOrObject::DestroyObject()
{
  MOZ_RELEASE_ASSERT(IsObject(), "Wrong type!");
  mValue.mObject.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrBooleanOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      if (value.isBoolean()) {
        done = (failed = !TrySetToBoolean(cx, value, tryNext)) || !tryNext;
        break;
      }
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
OwningStringOrBooleanOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrBooleanOrObject::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eBoolean: {
      DestroyBoolean();
      break;
    }
    case eObject: {
      DestroyObject();
      break;
    }
  }
}

bool
OwningStringOrBooleanOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eBoolean: {
      rval.setBoolean(mValue.mBoolean.Value());
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningStringOrBooleanOrObject::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eObject: {
      JS::TraceRoot(trc, &mValue.mObject.Value(), "mValue.mObject");
      break;
    }
    default: {
    }
  }
}

OwningStringOrBooleanOrObject&
OwningStringOrBooleanOrObject::operator=(OwningStringOrBooleanOrObject&& aOther)
{
  this->~OwningStringOrBooleanOrObject();
  new (this) OwningStringOrBooleanOrObject (std::move(aOther));
  return *this;
}



OwningStringOrMaybeSharedArrayBuffer::OwningStringOrMaybeSharedArrayBuffer(OwningStringOrMaybeSharedArrayBuffer&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
  }
}


bool
OwningStringOrMaybeSharedArrayBuffer::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrMaybeSharedArrayBuffer::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrMaybeSharedArrayBuffer::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrMaybeSharedArrayBuffer::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrMaybeSharedArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (DOMString or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
OwningStringOrMaybeSharedArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningStringOrMaybeSharedArrayBuffer::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningStringOrMaybeSharedArrayBuffer::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningStringOrMaybeSharedArrayBuffer::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrMaybeSharedArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
    return false;
  }
  return true;
}

bool
OwningStringOrMaybeSharedArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrMaybeSharedArrayBuffer::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
  }
}

bool
OwningStringOrMaybeSharedArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningStringOrMaybeSharedArrayBuffer::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningStringOrMaybeSharedArrayBuffer&
OwningStringOrMaybeSharedArrayBuffer::operator=(OwningStringOrMaybeSharedArrayBuffer&& aOther)
{
  this->~OwningStringOrMaybeSharedArrayBuffer();
  new (this) OwningStringOrMaybeSharedArrayBuffer (std::move(aOther));
  return *this;
}



OwningStringOrObject::OwningStringOrObject(OwningStringOrObject&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eObject: {
      mType = eObject;
      mValue.mObject.SetValue(std::move(aOther.mValue.mObject.Value()));
      break;
    }
  }
}


bool
OwningStringOrObject::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrObject::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrObject::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrObject::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}




[[nodiscard]] JSObject*&
OwningStringOrObject::RawSetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eObject;
  return mValue.mObject.SetValue();
}

[[nodiscard]] JSObject*&
OwningStringOrObject::SetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  Uninit();
  mType = eObject;
  return mValue.mObject.SetValue();
}


void
OwningStringOrObject::DestroyObject()
{
  MOZ_RELEASE_ASSERT(IsObject(), "Wrong type!");
  mValue.mObject.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
OwningStringOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrObject::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eObject: {
      DestroyObject();
      break;
    }
  }
}

bool
OwningStringOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningStringOrObject::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eObject: {
      JS::TraceRoot(trc, &mValue.mObject.Value(), "mValue.mObject");
      break;
    }
    default: {
    }
  }
}

OwningStringOrObject&
OwningStringOrObject::operator=(OwningStringOrObject&& aOther)
{
  this->~OwningStringOrObject();
  new (this) OwningStringOrObject (std::move(aOther));
  return *this;
}



OwningStringOrStringSequence::OwningStringOrStringSequence(OwningStringOrStringSequence&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eStringSequence: {
      mType = eStringSequence;
      mValue.mStringSequence.SetValue(std::move(aOther.mValue.mStringSequence.Value()));
      break;
    }
  }
}



bool
OwningStringOrStringSequence::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrStringSequence::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrStringSequence::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrStringSequence::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrStringSequence::TrySetToStringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Sequence<nsString>& memberSlot = RawSetAsStringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyStringSequence();
      tryNext = true;
      return true;
    }
    Sequence<nsString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
OwningStringOrStringSequence::TrySetToStringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToStringSequence(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Sequence<nsString>&
OwningStringOrStringSequence::RawSetAsStringSequence()
{
  if (mType == eStringSequence) {
    return mValue.mStringSequence.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eStringSequence;
  return mValue.mStringSequence.SetValue();
}

[[nodiscard]] Sequence<nsString>&
OwningStringOrStringSequence::SetAsStringSequence()
{
  if (mType == eStringSequence) {
    return mValue.mStringSequence.Value();
  }
  Uninit();
  mType = eStringSequence;
  return mValue.mStringSequence.SetValue();
}


void
OwningStringOrStringSequence::DestroyStringSequence()
{
  MOZ_RELEASE_ASSERT(IsStringSequence(), "Wrong type!");
  mValue.mStringSequence.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrStringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToStringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<DOMString>");
    return false;
  }
  return true;
}

bool
OwningStringOrStringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrStringSequence::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eStringSequence: {
      DestroyStringSequence();
      break;
    }
  }
}

bool
OwningStringOrStringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eStringSequence: {

      uint32_t length = mValue.mStringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!xpc::NonVoidStringToJsval(cx, mValue.mStringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningStringOrStringSequence&
OwningStringOrStringSequence::operator=(OwningStringOrStringSequence&& aOther)
{
  this->~OwningStringOrStringSequence();
  new (this) OwningStringOrStringSequence (std::move(aOther));
  return *this;
}


OwningStringOrStringSequence&
OwningStringOrStringSequence::operator=(const OwningStringOrStringSequence& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
    case eStringSequence: {
      SetAsStringSequence() = aOther.GetAsStringSequence();
      break;
    }
  }
  return *this;
}


OwningSupportedTypeOrObject::OwningSupportedTypeOrObject(OwningSupportedTypeOrObject&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eSupportedType: {
      mType = eSupportedType;
      mValue.mSupportedType.SetValue(std::move(aOther.mValue.mSupportedType.Value()));
      break;
    }
    case eObject: {
      mType = eObject;
      mValue.mObject.SetValue(std::move(aOther.mValue.mObject.Value()));
      break;
    }
  }
}


bool
OwningSupportedTypeOrObject::TrySetToSupportedType(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    SupportedType& memberSlot = RawSetAsSupportedType();
    {
      int index;
      if (!binding_detail::FindEnumStringIndex<true>(cx, value,
                                                                         binding_detail::EnumStrings<SupportedType>::Values,
                                                                         "SupportedType", "SupportedType branch of (SupportedType or object)",
                                                                         &index)) {
        return false;
      }
      MOZ_ASSERT(index >= 0);
      memberSlot = static_cast<SupportedType>(index);
    }
  }
  return true;
}

bool
OwningSupportedTypeOrObject::TrySetToSupportedType(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToSupportedType(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] SupportedType&
OwningSupportedTypeOrObject::RawSetAsSupportedType()
{
  if (mType == eSupportedType) {
    return mValue.mSupportedType.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eSupportedType;
  return mValue.mSupportedType.SetValue();
}

[[nodiscard]] SupportedType&
OwningSupportedTypeOrObject::SetAsSupportedType()
{
  if (mType == eSupportedType) {
    return mValue.mSupportedType.Value();
  }
  Uninit();
  mType = eSupportedType;
  return mValue.mSupportedType.SetValue();
}


void
OwningSupportedTypeOrObject::DestroySupportedType()
{
  MOZ_RELEASE_ASSERT(IsSupportedType(), "Wrong type!");
  mValue.mSupportedType.Destroy();
  mType = eUninitialized;
}




[[nodiscard]] JSObject*&
OwningSupportedTypeOrObject::RawSetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eObject;
  return mValue.mObject.SetValue();
}

[[nodiscard]] JSObject*&
OwningSupportedTypeOrObject::SetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  Uninit();
  mType = eObject;
  return mValue.mObject.SetValue();
}


void
OwningSupportedTypeOrObject::DestroyObject()
{
  MOZ_RELEASE_ASSERT(IsObject(), "Wrong type!");
  mValue.mObject.Destroy();
  mType = eUninitialized;
}



bool
OwningSupportedTypeOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      done = (failed = !TrySetToSupportedType(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
OwningSupportedTypeOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningSupportedTypeOrObject::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eSupportedType: {
      DestroySupportedType();
      break;
    }
    case eObject: {
      DestroyObject();
      break;
    }
  }
}

bool
OwningSupportedTypeOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eSupportedType: {
      if (!ToJSValue(cx, mValue.mSupportedType.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningSupportedTypeOrObject::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eObject: {
      JS::TraceRoot(trc, &mValue.mObject.Value(), "mValue.mObject");
      break;
    }
    default: {
    }
  }
}

OwningSupportedTypeOrObject&
OwningSupportedTypeOrObject::operator=(OwningSupportedTypeOrObject&& aOther)
{
  this->~OwningSupportedTypeOrObject();
  new (this) OwningSupportedTypeOrObject (std::move(aOther));
  return *this;
}



OwningTrustedHTMLOrNullIsEmptyString::OwningTrustedHTMLOrNullIsEmptyString(OwningTrustedHTMLOrNullIsEmptyString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      mType = eTrustedHTML;
      mValue.mTrustedHTML.SetValue(std::move(aOther.mValue.mTrustedHTML.Value()));
      break;
    }
    case eNullIsEmptyString: {
      mType = eNullIsEmptyString;
      mValue.mNullIsEmptyString.SetValue(std::move(aOther.mValue.mNullIsEmptyString.Value()));
      break;
    }
  }
}



bool
OwningTrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    static_assert(IsRefcounted<mozilla::dom::TrustedHTML>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrNullIsEmptyString::RawSetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrNullIsEmptyString::SetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  Uninit();
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}


void
OwningTrustedHTMLOrNullIsEmptyString::DestroyTrustedHTML()
{
  MOZ_RELEASE_ASSERT(IsTrustedHTML(), "Wrong type!");
  mValue.mTrustedHTML.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrNullIsEmptyString::TrySetToNullIsEmptyString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsNullIsEmptyString();
    if (!ConvertJSValueToString(cx, value, eEmpty, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrNullIsEmptyString::RawSetAsNullIsEmptyString()
{
  if (mType == eNullIsEmptyString) {
    return mValue.mNullIsEmptyString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eNullIsEmptyString;
  return mValue.mNullIsEmptyString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrNullIsEmptyString::SetAsNullIsEmptyString()
{
  if (mType == eNullIsEmptyString) {
    return mValue.mNullIsEmptyString.Value();
  }
  Uninit();
  mType = eNullIsEmptyString;
  return mValue.mNullIsEmptyString.SetValue();
}



void
OwningTrustedHTMLOrNullIsEmptyString::DestroyNullIsEmptyString()
{
  MOZ_RELEASE_ASSERT(IsNullIsEmptyString(), "Wrong type!");
  mValue.mNullIsEmptyString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrNullIsEmptyString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToNullIsEmptyString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
OwningTrustedHTMLOrNullIsEmptyString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedHTMLOrNullIsEmptyString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedHTML: {
      DestroyTrustedHTML();
      break;
    }
    case eNullIsEmptyString: {
      DestroyNullIsEmptyString();
      break;
    }
  }
}

bool
OwningTrustedHTMLOrNullIsEmptyString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eNullIsEmptyString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mNullIsEmptyString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedHTMLOrNullIsEmptyString&
OwningTrustedHTMLOrNullIsEmptyString::operator=(OwningTrustedHTMLOrNullIsEmptyString&& aOther)
{
  this->~OwningTrustedHTMLOrNullIsEmptyString();
  new (this) OwningTrustedHTMLOrNullIsEmptyString (std::move(aOther));
  return *this;
}


OwningTrustedHTMLOrNullIsEmptyString&
OwningTrustedHTMLOrNullIsEmptyString::operator=(const OwningTrustedHTMLOrNullIsEmptyString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      SetAsTrustedHTML() = aOther.GetAsTrustedHTML();
      break;
    }
    case eNullIsEmptyString: {
      SetAsNullIsEmptyString() = aOther.GetAsNullIsEmptyString();
      break;
    }
  }
  return *this;
}


OwningTrustedHTMLOrString::OwningTrustedHTMLOrString(OwningTrustedHTMLOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      mType = eTrustedHTML;
      mValue.mTrustedHTML.SetValue(std::move(aOther.mValue.mTrustedHTML.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningTrustedHTMLOrString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    static_assert(IsRefcounted<mozilla::dom::TrustedHTML>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedHTMLOrString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrString::RawSetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrString::SetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  Uninit();
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}


void
OwningTrustedHTMLOrString::DestroyTrustedHTML()
{
  MOZ_RELEASE_ASSERT(IsTrustedHTML(), "Wrong type!");
  mValue.mTrustedHTML.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningTrustedHTMLOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
OwningTrustedHTMLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedHTMLOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedHTML: {
      DestroyTrustedHTML();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningTrustedHTMLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedHTMLOrString&
OwningTrustedHTMLOrString::operator=(OwningTrustedHTMLOrString&& aOther)
{
  this->~OwningTrustedHTMLOrString();
  new (this) OwningTrustedHTMLOrString (std::move(aOther));
  return *this;
}


OwningTrustedHTMLOrString&
OwningTrustedHTMLOrString::operator=(const OwningTrustedHTMLOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      SetAsTrustedHTML() = aOther.GetAsTrustedHTML();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningTrustedScriptURLOrString::OwningTrustedScriptURLOrString(OwningTrustedScriptURLOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      mType = eTrustedScriptURL;
      mValue.mTrustedScriptURL.SetValue(std::move(aOther.mValue.mTrustedScriptURL.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningTrustedScriptURLOrString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    static_assert(IsRefcounted<mozilla::dom::TrustedScriptURL>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedScriptURLOrString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrString::RawSetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrString::SetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  Uninit();
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}


void
OwningTrustedScriptURLOrString::DestroyTrustedScriptURL()
{
  MOZ_RELEASE_ASSERT(IsTrustedScriptURL(), "Wrong type!");
  mValue.mTrustedScriptURL.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningTrustedScriptURLOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
OwningTrustedScriptURLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedScriptURLOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedScriptURL: {
      DestroyTrustedScriptURL();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningTrustedScriptURLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedScriptURLOrString&
OwningTrustedScriptURLOrString::operator=(OwningTrustedScriptURLOrString&& aOther)
{
  this->~OwningTrustedScriptURLOrString();
  new (this) OwningTrustedScriptURLOrString (std::move(aOther));
  return *this;
}


OwningTrustedScriptURLOrString&
OwningTrustedScriptURLOrString::operator=(const OwningTrustedScriptURLOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      SetAsTrustedScriptURL() = aOther.GetAsTrustedScriptURL();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningTrustedScriptURLOrUSVString::OwningTrustedScriptURLOrUSVString(OwningTrustedScriptURLOrUSVString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      mType = eTrustedScriptURL;
      mValue.mTrustedScriptURL.SetValue(std::move(aOther.mValue.mTrustedScriptURL.Value()));
      break;
    }
    case eUSVString: {
      mType = eUSVString;
      mValue.mUSVString.SetValue(std::move(aOther.mValue.mUSVString.Value()));
      break;
    }
  }
}



bool
OwningTrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    static_assert(IsRefcounted<mozilla::dom::TrustedScriptURL>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrUSVString::RawSetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrUSVString::SetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  Uninit();
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}


void
OwningTrustedScriptURLOrUSVString::DestroyTrustedScriptURL()
{
  MOZ_RELEASE_ASSERT(IsTrustedScriptURL(), "Wrong type!");
  mValue.mTrustedScriptURL.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrUSVString::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrUSVString::RawSetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrUSVString::SetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  Uninit();
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}



void
OwningTrustedScriptURLOrUSVString::DestroyUSVString()
{
  MOZ_RELEASE_ASSERT(IsUSVString(), "Wrong type!");
  mValue.mUSVString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrUSVString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
OwningTrustedScriptURLOrUSVString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedScriptURLOrUSVString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedScriptURL: {
      DestroyTrustedScriptURL();
      break;
    }
    case eUSVString: {
      DestroyUSVString();
      break;
    }
  }
}

bool
OwningTrustedScriptURLOrUSVString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedScriptURLOrUSVString&
OwningTrustedScriptURLOrUSVString::operator=(OwningTrustedScriptURLOrUSVString&& aOther)
{
  this->~OwningTrustedScriptURLOrUSVString();
  new (this) OwningTrustedScriptURLOrUSVString (std::move(aOther));
  return *this;
}


OwningTrustedScriptURLOrUSVString&
OwningTrustedScriptURLOrUSVString::operator=(const OwningTrustedScriptURLOrUSVString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      SetAsTrustedScriptURL() = aOther.GetAsTrustedScriptURL();
      break;
    }
    case eUSVString: {
      SetAsUSVString() = aOther.GetAsUSVString();
      break;
    }
  }
  return *this;
}


OwningUTF8StringOrArrayBuffer::OwningUTF8StringOrArrayBuffer(OwningUTF8StringOrArrayBuffer&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
  }
}


bool
OwningUTF8StringOrArrayBuffer::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrArrayBuffer::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrArrayBuffer::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrArrayBuffer::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrArrayBuffer::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (USVString or ArrayBuffer)");
      return false;
    }
  }
  return true;
}

bool
OwningUTF8StringOrArrayBuffer::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningUTF8StringOrArrayBuffer::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningUTF8StringOrArrayBuffer::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningUTF8StringOrArrayBuffer::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrArrayBuffer::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
    return false;
  }
  return true;
}

bool
OwningUTF8StringOrArrayBuffer::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrArrayBuffer::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
  }
}

bool
OwningUTF8StringOrArrayBuffer::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningUTF8StringOrArrayBuffer::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningUTF8StringOrArrayBuffer&
OwningUTF8StringOrArrayBuffer::operator=(OwningUTF8StringOrArrayBuffer&& aOther)
{
  this->~OwningUTF8StringOrArrayBuffer();
  new (this) OwningUTF8StringOrArrayBuffer (std::move(aOther));
  return *this;
}



OwningUTF8StringOrArrayBufferOrNull::OwningUTF8StringOrArrayBufferOrNull(OwningUTF8StringOrArrayBufferOrNull&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
  }
}




bool
OwningUTF8StringOrArrayBufferOrNull::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrArrayBufferOrNull::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrArrayBufferOrNull::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrArrayBufferOrNull::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrArrayBufferOrNull::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of (USVString or ArrayBuffer?)");
      return false;
    }
  }
  return true;
}

bool
OwningUTF8StringOrArrayBufferOrNull::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningUTF8StringOrArrayBufferOrNull::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningUTF8StringOrArrayBufferOrNull::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningUTF8StringOrArrayBufferOrNull::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrArrayBufferOrNull::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isNullOrUndefined()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (!done) {
      do {
        done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
        break;
      } while (false);
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBuffer");
      return false;
    }
  }
  return true;
}

bool
OwningUTF8StringOrArrayBufferOrNull::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrArrayBufferOrNull::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNull: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
  }
}

bool
OwningUTF8StringOrArrayBufferOrNull::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningUTF8StringOrArrayBufferOrNull::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningUTF8StringOrArrayBufferOrNull&
OwningUTF8StringOrArrayBufferOrNull::operator=(OwningUTF8StringOrArrayBufferOrNull&& aOther)
{
  this->~OwningUTF8StringOrArrayBufferOrNull();
  new (this) OwningUTF8StringOrArrayBufferOrNull (std::move(aOther));
  return *this;
}



OwningUTF8StringOrLong::OwningUTF8StringOrLong(OwningUTF8StringOrLong&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
  }
}



bool
OwningUTF8StringOrLong::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrLong::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrLong::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrLong::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (USVString or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningUTF8StringOrLong::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningUTF8StringOrLong::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningUTF8StringOrLong::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningUTF8StringOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrLong::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
  }
}

bool
OwningUTF8StringOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUTF8StringOrLong&
OwningUTF8StringOrLong::operator=(OwningUTF8StringOrLong&& aOther)
{
  this->~OwningUTF8StringOrLong();
  new (this) OwningUTF8StringOrLong (std::move(aOther));
  return *this;
}


OwningUTF8StringOrLong&
OwningUTF8StringOrLong::operator=(const OwningUTF8StringOrLong& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      SetAsUTF8String() = aOther.GetAsUTF8String();
      break;
    }
    case eLong: {
      SetAsLong() = aOther.GetAsLong();
      break;
    }
  }
  return *this;
}


OwningUTF8StringOrUTF8StringSequence::OwningUTF8StringOrUTF8StringSequence(OwningUTF8StringOrUTF8StringSequence&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eUTF8StringSequence: {
      mType = eUTF8StringSequence;
      mValue.mUTF8StringSequence.SetValue(std::move(aOther.mValue.mUTF8StringSequence.Value()));
      break;
    }
  }
}



bool
OwningUTF8StringOrUTF8StringSequence::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUTF8StringSequence::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUTF8StringSequence::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrUTF8StringSequence::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Sequence<nsCString>& memberSlot = RawSetAsUTF8StringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyUTF8StringSequence();
      tryNext = true;
      return true;
    }
    Sequence<nsCString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsCString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsCString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
OwningUTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Sequence<nsCString>&
OwningUTF8StringOrUTF8StringSequence::RawSetAsUTF8StringSequence()
{
  if (mType == eUTF8StringSequence) {
    return mValue.mUTF8StringSequence.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8StringSequence;
  return mValue.mUTF8StringSequence.SetValue();
}

[[nodiscard]] Sequence<nsCString>&
OwningUTF8StringOrUTF8StringSequence::SetAsUTF8StringSequence()
{
  if (mType == eUTF8StringSequence) {
    return mValue.mUTF8StringSequence.Value();
  }
  Uninit();
  mType = eUTF8StringSequence;
  return mValue.mUTF8StringSequence.SetValue();
}


void
OwningUTF8StringOrUTF8StringSequence::DestroyUTF8StringSequence()
{
  MOZ_RELEASE_ASSERT(IsUTF8StringSequence(), "Wrong type!");
  mValue.mUTF8StringSequence.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUTF8StringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<USVString>");
    return false;
  }
  return true;
}

bool
OwningUTF8StringOrUTF8StringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrUTF8StringSequence::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eUTF8StringSequence: {
      DestroyUTF8StringSequence();
      break;
    }
  }
}

bool
OwningUTF8StringOrUTF8StringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUTF8StringSequence: {

      uint32_t length = mValue.mUTF8StringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8StringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUTF8StringOrUTF8StringSequence&
OwningUTF8StringOrUTF8StringSequence::operator=(OwningUTF8StringOrUTF8StringSequence&& aOther)
{
  this->~OwningUTF8StringOrUTF8StringSequence();
  new (this) OwningUTF8StringOrUTF8StringSequence (std::move(aOther));
  return *this;
}


OwningUTF8StringOrUTF8StringSequence&
OwningUTF8StringOrUTF8StringSequence::operator=(const OwningUTF8StringOrUTF8StringSequence& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      SetAsUTF8String() = aOther.GetAsUTF8String();
      break;
    }
    case eUTF8StringSequence: {
      SetAsUTF8StringSequence() = aOther.GetAsUTF8StringSequence();
      break;
    }
  }
  return *this;
}


OwningUTF8StringOrUint8Array::OwningUTF8StringOrUint8Array(OwningUTF8StringOrUint8Array&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eUint8Array: {
      mType = eUint8Array;
      mValue.mUint8Array.SetValue(std::move(aOther.mValue.mUint8Array.Value()));
      break;
    }
  }
}


bool
OwningUTF8StringOrUint8Array::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUint8Array::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUint8Array::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrUint8Array::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUint8Array::TrySetToUint8Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Uint8Array& memberSlot = RawSetAsUint8Array();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyUint8Array();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
  }
  return true;
}

bool
OwningUTF8StringOrUint8Array::TrySetToUint8Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUint8Array(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Uint8Array&
OwningUTF8StringOrUint8Array::RawSetAsUint8Array()
{
  if (mType == eUint8Array) {
    return mValue.mUint8Array.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUint8Array;
  return mValue.mUint8Array.SetValue();
}

[[nodiscard]] Uint8Array&
OwningUTF8StringOrUint8Array::SetAsUint8Array()
{
  if (mType == eUint8Array) {
    return mValue.mUint8Array.Value();
  }
  Uninit();
  mType = eUint8Array;
  return mValue.mUint8Array.SetValue();
}


void
OwningUTF8StringOrUint8Array::DestroyUint8Array()
{
  MOZ_RELEASE_ASSERT(IsUint8Array(), "Wrong type!");
  mValue.mUint8Array.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUint8Array::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUint8Array(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Uint8Array");
    return false;
  }
  return true;
}

bool
OwningUTF8StringOrUint8Array::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrUint8Array::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eUint8Array: {
      DestroyUint8Array();
      break;
    }
  }
}

bool
OwningUTF8StringOrUint8Array::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUint8Array: {
      rval.setObject(*mValue.mUint8Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningUTF8StringOrUint8Array::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eUint8Array: {
      mValue.mUint8Array.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningUTF8StringOrUint8Array&
OwningUTF8StringOrUint8Array::operator=(OwningUTF8StringOrUint8Array&& aOther)
{
  this->~OwningUTF8StringOrUint8Array();
  new (this) OwningUTF8StringOrUint8Array (std::move(aOther));
  return *this;
}



OwningUndefinedOrCanvasPattern::OwningUndefinedOrCanvasPattern(OwningUndefinedOrCanvasPattern&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUndefined: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eUndefined;
      break;
    }
    case eCanvasPattern: {
      mType = eCanvasPattern;
      mValue.mCanvasPattern.SetValue(std::move(aOther.mValue.mCanvasPattern.Value()));
      break;
    }
  }
}





bool
OwningUndefinedOrCanvasPattern::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    static_assert(IsRefcounted<mozilla::dom::CanvasPattern>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningUndefinedOrCanvasPattern::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningUndefinedOrCanvasPattern::RawSetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningUndefinedOrCanvasPattern::SetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  Uninit();
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}


void
OwningUndefinedOrCanvasPattern::DestroyCanvasPattern()
{
  MOZ_RELEASE_ASSERT(IsCanvasPattern(), "Wrong type!");
  mValue.mCanvasPattern.Destroy();
  mType = eUninitialized;
}



bool
OwningUndefinedOrCanvasPattern::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isUndefined()) {
    SetUndefined();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern");
      return false;
    }
  }
  return true;
}

bool
OwningUndefinedOrCanvasPattern::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUndefinedOrCanvasPattern::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUndefined: {
      break;
    }
    case eCanvasPattern: {
      DestroyCanvasPattern();
      break;
    }
  }
}

bool
OwningUndefinedOrCanvasPattern::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUndefined: {
      rval.setUndefined();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUndefinedOrCanvasPattern&
OwningUndefinedOrCanvasPattern::operator=(OwningUndefinedOrCanvasPattern&& aOther)
{
  this->~OwningUndefinedOrCanvasPattern();
  new (this) OwningUndefinedOrCanvasPattern (std::move(aOther));
  return *this;
}


OwningUndefinedOrCanvasPattern&
OwningUndefinedOrCanvasPattern::operator=(const OwningUndefinedOrCanvasPattern& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUndefined: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eUndefined;
      break;
    }
    case eCanvasPattern: {
      SetAsCanvasPattern() = aOther.GetAsCanvasPattern();
      break;
    }
  }
  return *this;
}


OwningUndefinedOrCanvasPatternOrNull::OwningUndefinedOrCanvasPatternOrNull(OwningUndefinedOrCanvasPatternOrNull&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eUndefined: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eUndefined;
      break;
    }
    case eCanvasPattern: {
      mType = eCanvasPattern;
      mValue.mCanvasPattern.SetValue(std::move(aOther.mValue.mCanvasPattern.Value()));
      break;
    }
  }
}







bool
OwningUndefinedOrCanvasPatternOrNull::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    static_assert(IsRefcounted<mozilla::dom::CanvasPattern>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningUndefinedOrCanvasPatternOrNull::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningUndefinedOrCanvasPatternOrNull::RawSetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningUndefinedOrCanvasPatternOrNull::SetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  Uninit();
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}


void
OwningUndefinedOrCanvasPatternOrNull::DestroyCanvasPattern()
{
  MOZ_RELEASE_ASSERT(IsCanvasPattern(), "Wrong type!");
  mValue.mCanvasPattern.Destroy();
  mType = eUninitialized;
}



bool
OwningUndefinedOrCanvasPatternOrNull::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isUndefined()) {
    SetUndefined();
  } else if (value.isNull()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern");
      return false;
    }
  }
  return true;
}

bool
OwningUndefinedOrCanvasPatternOrNull::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUndefinedOrCanvasPatternOrNull::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNull: {
      break;
    }
    case eUndefined: {
      break;
    }
    case eCanvasPattern: {
      DestroyCanvasPattern();
      break;
    }
  }
}

bool
OwningUndefinedOrCanvasPatternOrNull::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eUndefined: {
      rval.setUndefined();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUndefinedOrCanvasPatternOrNull&
OwningUndefinedOrCanvasPatternOrNull::operator=(OwningUndefinedOrCanvasPatternOrNull&& aOther)
{
  this->~OwningUndefinedOrCanvasPatternOrNull();
  new (this) OwningUndefinedOrCanvasPatternOrNull (std::move(aOther));
  return *this;
}


OwningUndefinedOrCanvasPatternOrNull&
OwningUndefinedOrCanvasPatternOrNull::operator=(const OwningUndefinedOrCanvasPatternOrNull& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eUndefined: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eUndefined;
      break;
    }
    case eCanvasPattern: {
      SetAsCanvasPattern() = aOther.GetAsCanvasPattern();
      break;
    }
  }
  return *this;
}


OwningUndefinedOrNullOrCanvasPattern::OwningUndefinedOrNullOrCanvasPattern(OwningUndefinedOrNullOrCanvasPattern&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eUndefined: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eUndefined;
      break;
    }
    case eCanvasPattern: {
      mType = eCanvasPattern;
      mValue.mCanvasPattern.SetValue(std::move(aOther.mValue.mCanvasPattern.Value()));
      break;
    }
  }
}







bool
OwningUndefinedOrNullOrCanvasPattern::TrySetToCanvasPattern(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::CanvasPattern>& memberSlot = RawSetAsCanvasPattern();
    static_assert(IsRefcounted<mozilla::dom::CanvasPattern>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::CanvasPattern, mozilla::dom::CanvasPattern>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyCanvasPattern();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningUndefinedOrNullOrCanvasPattern::TrySetToCanvasPattern(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningUndefinedOrNullOrCanvasPattern::RawSetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::CanvasPattern>&
OwningUndefinedOrNullOrCanvasPattern::SetAsCanvasPattern()
{
  if (mType == eCanvasPattern) {
    return mValue.mCanvasPattern.Value();
  }
  Uninit();
  mType = eCanvasPattern;
  return mValue.mCanvasPattern.SetValue();
}


void
OwningUndefinedOrNullOrCanvasPattern::DestroyCanvasPattern()
{
  MOZ_RELEASE_ASSERT(IsCanvasPattern(), "Wrong type!");
  mValue.mCanvasPattern.Destroy();
  mType = eUninitialized;
}



bool
OwningUndefinedOrNullOrCanvasPattern::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  if (value.isUndefined()) {
    SetUndefined();
  } else if (value.isNull()) {
    SetNull();
  } else {
    bool done = false, failed = false, tryNext;
    if (value.isObject()) {
      done = (failed = !TrySetToCanvasPattern(cx, value, tryNext, passedToJSImpl)) || !tryNext;
    }
    if (failed) {
      return false;
    }
    if (!done) {
      cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "CanvasPattern");
      return false;
    }
  }
  return true;
}

bool
OwningUndefinedOrNullOrCanvasPattern::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUndefinedOrNullOrCanvasPattern::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNull: {
      break;
    }
    case eUndefined: {
      break;
    }
    case eCanvasPattern: {
      DestroyCanvasPattern();
      break;
    }
  }
}

bool
OwningUndefinedOrNullOrCanvasPattern::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNull: {
      rval.setNull();
      return true;
    }
    case eUndefined: {
      rval.setUndefined();
      return true;
    }
    case eCanvasPattern: {
      if (!GetOrCreateDOMReflector(cx, mValue.mCanvasPattern.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUndefinedOrNullOrCanvasPattern&
OwningUndefinedOrNullOrCanvasPattern::operator=(OwningUndefinedOrNullOrCanvasPattern&& aOther)
{
  this->~OwningUndefinedOrNullOrCanvasPattern();
  new (this) OwningUndefinedOrNullOrCanvasPattern (std::move(aOther));
  return *this;
}


OwningUndefinedOrNullOrCanvasPattern&
OwningUndefinedOrNullOrCanvasPattern::operator=(const OwningUndefinedOrNullOrCanvasPattern& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNull: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eNull;
      break;
    }
    case eUndefined: {
      MOZ_ASSERT(mType == eUninitialized);
      mType = eUndefined;
      break;
    }
    case eCanvasPattern: {
      SetAsCanvasPattern() = aOther.GetAsCanvasPattern();
      break;
    }
  }
  return *this;
}


OwningUnrestrictedDoubleOrString::OwningUnrestrictedDoubleOrString(OwningUnrestrictedDoubleOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUnrestrictedDouble: {
      mType = eUnrestrictedDouble;
      mValue.mUnrestrictedDouble.SetValue(std::move(aOther.mValue.mUnrestrictedDouble.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningUnrestrictedDoubleOrString::TrySetToUnrestrictedDouble(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    double& memberSlot = RawSetAsUnrestrictedDouble();
    if (!ValueToPrimitive<double, eDefault>(cx, value, "Unrestricted double branch of (unrestricted double or DOMString)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] double&
OwningUnrestrictedDoubleOrString::RawSetAsUnrestrictedDouble()
{
  if (mType == eUnrestrictedDouble) {
    return mValue.mUnrestrictedDouble.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUnrestrictedDouble;
  return mValue.mUnrestrictedDouble.SetValue();
}

[[nodiscard]] double&
OwningUnrestrictedDoubleOrString::SetAsUnrestrictedDouble()
{
  if (mType == eUnrestrictedDouble) {
    return mValue.mUnrestrictedDouble.Value();
  }
  Uninit();
  mType = eUnrestrictedDouble;
  return mValue.mUnrestrictedDouble.SetValue();
}


void
OwningUnrestrictedDoubleOrString::DestroyUnrestrictedDouble()
{
  MOZ_RELEASE_ASSERT(IsUnrestrictedDouble(), "Wrong type!");
  mValue.mUnrestrictedDouble.Destroy();
  mType = eUninitialized;
}



bool
OwningUnrestrictedDoubleOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningUnrestrictedDoubleOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningUnrestrictedDoubleOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningUnrestrictedDoubleOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningUnrestrictedDoubleOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToUnrestrictedDouble(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningUnrestrictedDoubleOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUnrestrictedDoubleOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUnrestrictedDouble: {
      DestroyUnrestrictedDouble();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningUnrestrictedDoubleOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUnrestrictedDouble: {
      rval.set(JS_NumberValue(double(mValue.mUnrestrictedDouble.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUnrestrictedDoubleOrString&
OwningUnrestrictedDoubleOrString::operator=(OwningUnrestrictedDoubleOrString&& aOther)
{
  this->~OwningUnrestrictedDoubleOrString();
  new (this) OwningUnrestrictedDoubleOrString (std::move(aOther));
  return *this;
}


OwningUnrestrictedDoubleOrString&
OwningUnrestrictedDoubleOrString::operator=(const OwningUnrestrictedDoubleOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUnrestrictedDouble: {
      SetAsUnrestrictedDouble() = aOther.GetAsUnrestrictedDouble();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningUnrestrictedFloatOrString::OwningUnrestrictedFloatOrString(OwningUnrestrictedFloatOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUnrestrictedFloat: {
      mType = eUnrestrictedFloat;
      mValue.mUnrestrictedFloat.SetValue(std::move(aOther.mValue.mUnrestrictedFloat.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningUnrestrictedFloatOrString::TrySetToUnrestrictedFloat(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    float& memberSlot = RawSetAsUnrestrictedFloat();
    if (!ValueToPrimitive<float, eDefault>(cx, value, "Unrestricted float branch of (unrestricted float or DOMString)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] float&
OwningUnrestrictedFloatOrString::RawSetAsUnrestrictedFloat()
{
  if (mType == eUnrestrictedFloat) {
    return mValue.mUnrestrictedFloat.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUnrestrictedFloat;
  return mValue.mUnrestrictedFloat.SetValue();
}

[[nodiscard]] float&
OwningUnrestrictedFloatOrString::SetAsUnrestrictedFloat()
{
  if (mType == eUnrestrictedFloat) {
    return mValue.mUnrestrictedFloat.Value();
  }
  Uninit();
  mType = eUnrestrictedFloat;
  return mValue.mUnrestrictedFloat.SetValue();
}


void
OwningUnrestrictedFloatOrString::DestroyUnrestrictedFloat()
{
  MOZ_RELEASE_ASSERT(IsUnrestrictedFloat(), "Wrong type!");
  mValue.mUnrestrictedFloat.Destroy();
  mType = eUninitialized;
}



bool
OwningUnrestrictedFloatOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningUnrestrictedFloatOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningUnrestrictedFloatOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningUnrestrictedFloatOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningUnrestrictedFloatOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  do {
    if (value.isNumber()) {
      done = (failed = !TrySetToUnrestrictedFloat(cx, value, tryNext)) || !tryNext;
      break;
    }
    done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
    break;
  } while (false);
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "");
    return false;
  }
  return true;
}

bool
OwningUnrestrictedFloatOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUnrestrictedFloatOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUnrestrictedFloat: {
      DestroyUnrestrictedFloat();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningUnrestrictedFloatOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUnrestrictedFloat: {
      rval.set(JS_NumberValue(double(mValue.mUnrestrictedFloat.Value())));
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUnrestrictedFloatOrString&
OwningUnrestrictedFloatOrString::operator=(OwningUnrestrictedFloatOrString&& aOther)
{
  this->~OwningUnrestrictedFloatOrString();
  new (this) OwningUnrestrictedFloatOrString (std::move(aOther));
  return *this;
}


OwningUnrestrictedFloatOrString&
OwningUnrestrictedFloatOrString::operator=(const OwningUnrestrictedFloatOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUnrestrictedFloat: {
      SetAsUnrestrictedFloat() = aOther.GetAsUnrestrictedFloat();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}
} // namespace mozilla::dom

