use crate::gobject_models::GSidebarSelection;
use crate::{content_page::ArticleListMode, settings::SettingsError};
use serde::{Deserialize, Serialize};

static JSON_NAME: &str = "content_page_state.json";

#[derive(Clone, Debug, Serialize, Deserialize)]
pub struct SavedState {
    pub sidebar_selection: GSidebarSelection,
    pub article_list_mode: ArticleListMode,
    pub search_term: String,
    pub prefer_scraped_content: bool,
    pub selected_article: Option<(u32, String)>,
    pub window_size: (i32, i32),
    pub maximized: bool,
    pub article_view_visible: bool,
    pub sidebar_visible: bool,
    pub article_view_zoom: f64,
    pub subscriptions_expanded: bool,
    pub tags_expanded: bool,
    pub expanded_categories: Vec<u32>,
    pub volume: f64,
}

impl Default for SavedState {
    fn default() -> Self {
        Self {
            sidebar_selection: GSidebarSelection::all(0),
            article_list_mode: ArticleListMode::All,
            search_term: String::new(),
            prefer_scraped_content: false,
            selected_article: None,
            window_size: (1200, 650),
            maximized: false,
            article_view_visible: true,
            sidebar_visible: true,
            article_view_zoom: 1.0,
            subscriptions_expanded: true,
            tags_expanded: true,
            expanded_categories: vec![],
            volume: 1.0,
        }
    }
}

impl SavedState {
    pub fn new_from_file() -> Result<Self, SettingsError> {
        let path = crate::app::STATE_DIR.join(JSON_NAME);
        let data = match std::fs::read_to_string(path) {
            Ok(data) => data,
            Err(_) => {
                let fallback_path = crate::app::CONFIG_DIR.join(JSON_NAME);
                std::fs::read_to_string(fallback_path)?
            }
        };
        let state = serde_json::from_str::<Self>(&data)?;
        Ok(state)
    }

    pub fn write(&self) -> Result<(), SettingsError> {
        if !crate::app::STATE_DIR.exists() {
            std::fs::create_dir_all(crate::app::STATE_DIR.as_path())?;
        }
        let path = crate::app::STATE_DIR.join(JSON_NAME);
        let data = serde_json::to_string_pretty(self)?;
        std::fs::write(path, data)?;
        Ok(())
    }
}
